import { Injectable, Logger } from '@nestjs/common';
import { AssetDescriptor } from '../dtos/asset-descriptor';
import { ConfigService } from '@nestjs/config';
import { AssetHashingTrxInput } from '../dtos/asset-hashing-trxin';
import { BlockchainService } from '../integration/blockchain-service';
import { GovService } from '../integration/gov-service';
import { FdacService } from '../integration/fdac-service';
import { ApmService } from '../integration/apm-service';
import { TmService } from '../integration/tm-service';
import * as short from 'short-uuid';
import { UserInfo } from 'src/auth/user-info';
import { AssetMetadata } from 'src/dtos/asset-metadata';
import { ErrorCondition, getISOTimeFromString, isApmDisabled, isAuthDisabled, isFdacDisabled, isInternalApi, isPrivilegedUser, isRqueueDisabled, safeParseInt, throwExceptionWithErrorCondition, checkPublishingRules, isXcheckDisabled, isTmDisabled } from 'src/utils/generic-utils';
import { MarkdownUtils } from 'src/utils/markdown-utils';
import { AssetCatalogueTypes, AssetType, FAME_PFX } from 'src/constants';
import { OfferingCatalogue } from '../offerings/offering-catalogue';
import { OfferingService } from '../offerings/offering-service';
import { AssetRepublishJobState } from './asset-republish-job-state';
import { OfferingRecord } from 'src/dtos/offering-record';


@Injectable()
export class AssetService {

    private readonly skipAuth: boolean = false;
    private readonly skipFdac: boolean = false;
    private readonly skipApm: boolean = false;
    private readonly skipRQueue: boolean = false;
    private readonly skipXCheck: boolean = false;
    private readonly skipTM: boolean = false;

    constructor(
        private readonly config: ConfigService,
        private readonly blockchain: BlockchainService,
        private readonly rqueue: GovService,
        private readonly fdac: FdacService,
        private readonly apm: ApmService,
        private readonly ocat: OfferingCatalogue,
        private readonly offeringService: OfferingService,
        private readonly tm: TmService,
        private readonly jobState: AssetRepublishJobState,
    ) {
        this.skipAuth = isAuthDisabled(this.config);
        this.skipFdac = isFdacDisabled(this.config);
        this.skipApm = isApmDisabled(this.config);
        this.skipRQueue = isRqueueDisabled(this.config);
        this.skipXCheck = isXcheckDisabled(this.config);
        this.skipTM = isTmDisabled(this.config);

        // We only subscribe to blockchain events if we are running
        // the Open API service, otherwise we will execute the same
        // event handling logic twice! The Internal API service is
        // not going to submit any asset-related blockchain transaction.
        if (!isInternalApi(config)) {
            this.subscribeToContractEvents();
        }
    }

    public async publishAsset(user: UserInfo, asset: AssetDescriptor): Promise<AssetHashingTrxInput> {
        // Prepare the asset descriptor: translate type, set creator, process Markdown
        this.prepareAssetDescriptor(user, asset);

        // Create the catalogue entry and get AID and hash
        const { aid, hash } = await this.createCatalogueEntry(asset);

        // Create default policy for the asset
        await this.createAssetPolicy(aid, asset.dcterms_type, user);

        // Add user request to the queue
        this.enqueueAssetRequest(aid, user, 'publishing');

        Logger.log('First phase of publishing for asset ' + aid + ' completed');

        // Build and return the transaction response
        return this.buildTransactionResponse(aid, hash, 'postAsset');
    }

    public async republishAsset(user: UserInfo, aid: string, asset: AssetDescriptor): Promise<AssetHashingTrxInput> {
        // Store the original AID for lineage tracking
        const predecessorAid = aid;

        // CRITICAL: Retrieve ACTIVE offerings BEFORE unpublishing the asset
        // We need to capture the offerings that are active at the time of republishing,
        // not those that were already archived by the user
        Logger.debug(`Retrieving active offerings for asset ${aid} before unpublishing`);
        const activeOfferingsToClone: OfferingRecord[] = await this.ocat.listActiveOfferingsByAsset(aid);
        Logger.debug(`Found ${activeOfferingsToClone.length} active offerings to be cloned for asset ${aid}`);

        // Unpublish the existing asset and get hold of the predecessor's catalogue entry (DCAT descriptor)
        Logger.debug(`Unpublishing existing asset ${aid} as part of revision process`);
        const predecessor = await this.unpublishAsset(user, aid);

        // Fix the asset descriptor passed by the caller: replicate title and type, set creator, process Markdown
        this.prepareAssetDescriptor(user, asset, predecessor);

        // Create the catalogue entry and get new AID and hash
        Logger.debug(`Publishing new version to replace asset ${aid}`);
        const { aid: newAid, hash } = await this.createCatalogueEntry(asset);

        // Clone the policy of the predecessor for the revised asset
        await this.apm.clonePolicy(predecessorAid, newAid, user);

        // Add user request to the queue
        this.enqueueAssetRequest(newAid, user, 'republishing');

        // Start background job to clone offerings from the old asset to the new one
        // This is done asynchronously to avoid keeping the user waiting
        // Pass the offerings we captured before unpublishing
        if (activeOfferingsToClone.length > 0) {
            Logger.debug(`Starting offering cloning job for asset ${newAid}`);
            this.startOfferingCloningJob(predecessorAid, newAid, user, activeOfferingsToClone);
        }

        Logger.log(`First phase of republishing completed - original AID: ${predecessorAid}, new AID: ${newAid}`);

        // Build and return the transaction response with lineage tracking
        return this.buildTransactionResponse(newAid, hash, 'postAsset', predecessorAid);
    }

    public async unpublishAsset(user: UserInfo, aid: string): Promise<any> {
        let target = null;

        // Step 1: Verify that the asset exists and user has permission to revise it
        if (this.skipXCheck || this.skipFdac) {
            Logger.debug('FDAC and/or asset x-check are disabled, skipping all asset rules verification');
        } else {
            target = await checkPublishingRules(this.fdac, this.apm, null, aid, user, this.skipAuth);
        }

        // Step 2: archive all linked offerings, if any
        try {
            const archivedCount = await this.ocat.setArchivedByAsset(aid);
            Logger.log(`Asset ${aid} unpublishing: offering status set to ARCHIVED on ${archivedCount} linked offerings`);
        } catch (err) {
            Logger.error(`Error archiving offerings for asset ${aid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not unpublish asset due to database error');
        }

        // Step 3: archive the asset in the catalogue
        if (this.skipApm) {
            Logger.debug('APM is disabled, cannot set asset status to ARCHIVED');
        } else {
            try {
                await this.apm.setArchived(aid, user);
                Logger.log(`Asset ${aid} unpublishing: asset status set to ARCHIVED in APM`);
            } catch (err) {
                Logger.error(`Error calling APM to archive asset ${aid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error communicating with the APM module');
            }
        }

        // Return the unpublished asset's catalogue entry (DCAT descriptor)
        return target;
    }

    public async retrieveAssetData(user: UserInfo, aid: string): Promise<AssetDescriptor> {
        if (this.skipFdac) {
            Logger.debug('FDAC is disabled, cannot retrieve the catalogue entry for asset with AID ' + aid);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
        }

        let entry: any = await this.getCatalogueDcatEntry(aid); // exception if not found

        if (this.skipApm) {
            Logger.debug('APM is disabled, no policies are applied to asset with AID ' + aid);
        } else if (!isPrivilegedUser(user)) {
            if (!await this.apm.checkVisibility(aid, user)) {
                Logger.debug('Access to asset with AID ' + aid + ' denied to user with UID ' + user.uid);
                throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
            }
        }

        return entry;
    }

    public async retrieveTracingData(user: UserInfo, aid: string): Promise<AssetMetadata> {
        if (this.skipFdac) {
            Logger.debug('FDAC is disabled, cannot retrieve the catalogue entry for asset with AID ' + aid);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
        }

        let entry: any = await this.getCatalogueCanonicalEntry(aid); // exception if not found

        if (this.skipApm) {
            Logger.debug('APM is disabled, no policies are applied to asset with AID ' + aid);
        } else if (!isPrivilegedUser(user)) {
            if (!await this.apm.checkVisibility(aid, user)) {
                Logger.debug('Access to asset with AID ' + aid + ' denied to user with UID ' + user.uid);
                throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
            }
        }

        let rec: any = null;
        try {
            rec = await this.blockchain.tracingContract.methods.getAsset(aid).call();
        } catch (err) {
            Logger.error('Error calling Tracing contract method getAsset(): ', err);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error calling the Tracing Ledger contract');
        }

        const meta: AssetMetadata = new AssetMetadata();
        meta.aid = aid;
        meta.entry = entry;
        if (safeParseInt(rec[3]) > 0) { // if the "registered" timestamp is set to a positive number, the record exists on the ledger
            meta.hash = rec[1];
            if (isPrivilegedUser(user)) { // TID is hidden from regular users
                meta.tid = rec[2];
            } else {
                meta.tid = 'N/A';
            }
            meta.registered = getISOTimeFromString(rec[3]);
            const entryHash: string = this.getHashFromCatalogueEntry(entry);
            meta.trustworthy = (entryHash === meta.hash);

            // Get lineage information
            try {
                const lineageArray: string[] = await this.blockchain.tracingContract.methods.getLineage(aid).call();
                if (lineageArray && lineageArray.length > 1) {
                    const predecessors = lineageArray.slice(0, -1); // Exclude last element (current asset)
                    meta.lineage = predecessors.join(',');
                }
            } catch (err) {
                Logger.error('Error calling Tracing contract method getLineage(): ', err);
                // Don't throw an error, just continue without lineage information
            }
        } else {
            meta.hash = 'N/A';
            meta.tid = 'N/A';
            meta.registered = 'N/A';
            meta.trustworthy = false;
        }
        return meta;
    }

    // subscribe to AssetEvent events emitted by the postAsset() and rehashAsset() contract methods
    private async subscribeToContractEvents() {
        const subscription = await this.blockchain.getTracingEventsSubscription();
        subscription.on('data', (log: any) => {
            try {
                const eventObj = this.blockchain.decodeTracingLogs(log);
                const aid: string = eventObj.aid;
                const action: string = eventObj.action;
                Logger.log('Transaction confirmed for asset with AID ' + aid + ", action is " + action);
                this.finalizeCatalogueEntry(aid, action);
            } catch (err) {
                // this is unlikely to happen, but we need to catch it to avoid crashing the process
                Logger.error('Error in Tracing contract event handling: ', err);
            }
        });
        subscription.on('error', (error: Error) => {
            Logger.error('Error in Tracing contract event subscription: ', error);
            // TODO uncler how to handle this condition, as we probably don't have access to the AID
        });
    }

    private translateAssetType(type: string): string {
        const index = Object.values(AssetType).indexOf(type as AssetType);
        if (index !== -1) {
            return AssetCatalogueTypes[index];
        } else {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Asset type ' + type + ' is not recognized as a valid type');
        }
    }

    private async getCatalogueDcatEntry(aid: string): Promise<any> {
        const entry: any = await this.fdac.getAssetDcatDescription(aid);
        if (entry) {
            return entry;
        } else {
            Logger.error(`Error calling the FDAC module: catalogue entry with AID ${aid} could not be retrieved`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
        }
    }

    private async getCatalogueCanonicalEntry(aid: string): Promise<any> {
        const entry: any = await this.fdac.getAssetCanonicalDescription(aid);
        if (entry) {
            return entry;
        } else {
            Logger.error(`Error calling the FDAC module: catalogue entry with AID ${aid} could not be retrieved`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
        }
    }

    private getHashFromCatalogueEntry(entry: any): string {
        return this.blockchain.getCanonicalHash(JSON.stringify(entry));
    }

    private finalizeCatalogueEntry(aid: string, action: string) {
        if (this.skipFdac) {
            Logger.debug('FDAC is disabled');
        } else {
            let messagePfx = null;
            if (action === 'REGISTERED') {
                messagePfx = 'Second phase of publishing for asset with AID ' + aid;
            } else {
                Logger.warn('Unsupported transaction type for asset with AID ' + aid + ", action is " + action);
                return;
            }

            // we use the Promise interface to ensure that exceptions are not propagated
            // to the blockchain event handling mechanism, as this would crash the Node process
            this.fdac.certifyEntry(aid)
                .then(() => {
                    Logger.log(messagePfx + ' completed');
                    this.finalizeUserRequest(aid, true);
                })
                .catch((error) => {
                    Logger.error(messagePfx + ' FAILED: ', error);
                    this.finalizeUserRequest(aid, false);
                });
        }
    }

    private finalizeUserRequest(aid: string, success: boolean) {
        if (this.skipAuth || this.skipRQueue) {
            Logger.debug('Authentication and/or RQUEUE are disabled: no user request management');
        } else {
            try {
                success ? this.rqueue.confirmRequest(aid, 'Asset was published with AID ' + aid) :
                    this.rqueue.rejectRequest(aid, 'Asset publishing failed');
            } catch (err) {
                // should never happen as exceptions are managed by the queue service
                Logger.error(`Error updating user request with id ${aid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
        }
    }

    private prepareAssetDescriptor(user: UserInfo, asset: AssetDescriptor, predecessor?: any): void {
        if (predecessor) {
            // Title and type cannot be changed in a revision
            asset.dcterms_title = predecessor.dcterms_title;
            asset.dcterms_type = predecessor.dcterms_type;
        } else {
            // Translate the asset type from AssetType (user input) to AssetCatalogueTypes (FDAC)
            asset.dcterms_type = this.translateAssetType(asset.dcterms_type); // throws exception if translation fails
        }

        // Set the user affiliation as the creator of the asset
        if (this.skipAuth) {
            Logger.debug('AAI is disabled: user affiliation is unavailable and asset provenance is faked');
            asset.dcterms_creator = 'fake.com';
        } else {
            if (!user.affiliation) {
                Logger.error(`Asset publisher has no affiliation`);
                throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Asset publisher has no affiliation');
            }
            asset.dcterms_creator = FAME_PFX + user.affiliation;
        }
        
        // Process Markdown fields: convert safe Markdown to HTML
        if (asset.dcterms_description_long) {
            asset.dcterms_description_long = MarkdownUtils.markdownToHtml(asset.dcterms_description_long);
        }
    }

    private async createCatalogueEntry(asset: AssetDescriptor): Promise<{ aid: string; hash: string }> {
        let aid: string = null;
        let hash: string = null;

        if (this.skipFdac) {
            // simulating the catalogue
            aid = short.generate(); // AID is generated randomly
            hash = this.blockchain.getCanonicalHash(JSON.stringify(asset)); // hashing the user's input
            Logger.debug(`FDAC is disabled, asset catalogue creation is simulated - AID: ${aid}, hash value: ${hash}`);
        } else {
            // create the catalogue entry for the asset
            Logger.debug('Creating asset catalogue entry via FDAC module: ' + JSON.stringify(asset));
            try {
                aid = await this.fdac.createEntry(asset); // ID is assigned by FDAC
                if (aid) {
                    // retrieve the normalized descriptor back from the FDAC: we calculate the
                    // entry's hash value from that (entry), not from the user's input (asset)
                    hash = this.getHashFromCatalogueEntry(await this.getCatalogueCanonicalEntry(aid));
                } else {
                    Logger.error(`Error calling the FDAC module for creating a new catalogue entry: no AID assigned`);
                    throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error calling the FDAC module');
                }
            } catch (err) {
                Logger.error(`Error calling the FDAC module for creating asset catalogue entry: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error communicating with the FDAC module');
            }
        }

        return { aid, hash };
    }

    private async createAssetPolicy(aid: string, assetType: string, user: UserInfo): Promise<void> {
        if (this.skipAuth || this.skipFdac || this.skipApm) {
            Logger.debug('AAI, FDAC and/or APM are disabled, no default policy for asset with AID ' + aid + ' has been created');
        } else {
            try {
                // create a default policy for the asset
                await this.apm.createPolicy(aid, assetType, user);
                Logger.debug('Default policy for asset ' + aid + ' has been created');
            } catch (err) {
                Logger.error(`Error calling the APM module for setting the policy of asset ${aid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error communicating with the APM module');
            }
        }
    }

    private enqueueAssetRequest(aid: string, user: UserInfo, operation: string): void {
        if (this.skipAuth || this.skipRQueue) {
            Logger.debug('Authentication and/or RQUEUE are disabled: no user request management');
        } else {
            try {
                this.rqueue.enqueueRequest(aid, user.uid, '', `Processing asset ${operation} request`);
            } catch (err) {
                // should never happen as exceptions are managed by the queue service, anyway we swallow...
                Logger.error(`Error adding user request to the queue: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
        }
    }

    private buildTransactionResponse(aid: string, hash: string, functionName: string, predecessorAid?: string): AssetHashingTrxInput {
        let resp = new AssetHashingTrxInput();
        resp.aid = aid;
        resp.hash = hash;
        resp.functionName = functionName;
        if (predecessorAid) {
            resp.predecessorAid = predecessorAid;
        }
        resp = this.blockchain.annotateTracingTrxInput(resp); // adds contract address and abi
        return resp;
    }

    /**
     * Starts a background job to clone offerings from the old asset to the new one.
     * This method returns immediately and the cloning process continues asynchronously.
     * @param activeOfferings The list of offerings that were ACTIVE before the asset was unpublished
     */
    private async startOfferingCloningJob(predecessorAid: string, newAssetAid: string, user: UserInfo, activeOfferings: OfferingRecord[]): Promise<void> {
        try {
            if (activeOfferings.length === 0) {
                Logger.log(`No active offerings to clone for asset ${predecessorAid}`);
                return;
            }

            const offeringOids = activeOfferings.map(o => o.oid);
            Logger.log(`Starting background job to clone ${offeringOids.length} offerings from asset ${predecessorAid} to ${newAssetAid}`);

            // Create the background job
            this.jobState.createJob(newAssetAid, predecessorAid, offeringOids);

            // Start processing offerings asynchronously
            // We use setImmediate to ensure we return control to the caller immediately
            setImmediate(() => this.processNextOffering(newAssetAid, user));
        } catch (err) {
            Logger.error(`Error starting offering cloning job for asset ${newAssetAid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            // We don't throw here because the asset republishing itself was successful
            // The offering cloning is a "nice to have" feature that shouldn't block the main flow
        }
    }

    /**
     * Processes the next offering in the cloning job queue.
     * This method calls itself recursively until all offerings are processed.
     */
    private async processNextOffering(jobId: string, user: UserInfo): Promise<void> {
        const oid = this.jobState.getNextOffering(jobId);

        if (!oid) {
            // No more offerings to process
            return;
        }

        try {
            await this.cloneOffering(oid, jobId, user);
            this.jobState.markOfferingCompleted(jobId, oid);
        } catch (err) {
            Logger.error(`Error cloning offering ${oid} for job ${jobId}: ${err.name}, Message: ${err.message}`);
            this.jobState.markOfferingFailed(jobId, oid, err.message || 'Unknown error');
        }

        // Process the next offering
        setImmediate(() => this.processNextOffering(jobId, user));
    }

    /**
     * Clones a single offering, following the same multi-step process as publishOffering.
     * The offering is created as a DRAFT and then confirmed asynchronously by the TM service.
     * This method does NOT throw exceptions - errors are logged and null is returned on failure.
     */
    private async cloneOffering(originalOid: string, newAssetAid: string, user: UserInfo): Promise<void> {
        Logger.debug(`Cloning offering ${originalOid} for new asset ${newAssetAid}`);

        // Retrieve the original offering
        const originalOffering: OfferingRecord = await this.ocat.retrieve(originalOid);
        if (!originalOffering) {
            Logger.error(`Original offering ${originalOid} not found in catalogue - cannot clone`);
            throw new Error(`Original offering ${originalOid} not found in catalogue`);
        }

        // Create a new offering descriptor with the same properties but pointing to the new asset
        const newOfferingDescriptor = Object.assign({}, originalOffering.offering);
        newOfferingDescriptor.asset = newAssetAid;

        Logger.debug(`Publishing cloned offering for asset ${newAssetAid} based on original offering ${originalOid}`);

        // Use the OfferingService.publishOffering with isCloning=true
        // When isCloning=true:
        // 1. Does NOT convert Markdown (offering already has HTML)
        // 2. Does NOT throw exceptions (returns null on error)
        // 3. Does NOT manage user request queue
        // 4. Creates a DRAFT entry in the catalogue
        // 5. Calls the TM service to set up trading
        // 6. TM service will asynchronously call back to confirm the offering
        const result = await this.offeringService.publishOffering(user, newOfferingDescriptor, undefined, true);

        if (result.id) {
            Logger.debug(`Cloned offering ${result.id} created successfully for asset ${newAssetAid} based on ${originalOid}`);
        } else {
            Logger.error(`Failed to clone offering ${originalOid} for asset ${newAssetAid}`);
            throw new Error(`Failed to clone offering ${originalOid}`);
        }
    }
}
