import {
    CanActivate,
    ExecutionContext,
    Injectable,
    Logger,
    UnauthorizedException,
} from "@nestjs/common";
import { ConfigService } from "@nestjs/config";
import axios from "axios";
import { Request } from "express";
import { isAuthDisabled } from 'src/utils/generic-utils';

@Injectable()
export class AuthGuard implements CanActivate {

    private readonly skipAuth: boolean = false;
    private readonly url: string = 'http://fake-auth-service.com';

    constructor(private configService: ConfigService) {
        this.skipAuth = isAuthDisabled(this.configService);
        this.url = this.configService.get<string>('AUTH_URL');
    }

    async canActivate(context: ExecutionContext): Promise<boolean> {
        if (this.skipAuth) {
            Logger.debug('User authentication is disabled');
            return true;
        }

        const request = context.switchToHttp().getRequest();
        const token = this.extractTokenFromHeader(request);
        if (!token) {
            throw new UnauthorizedException('Authentication token missing');
        }

        let response = null;
        try {
            let body = {
                jwtToken: token
            };
            const config = {
                headers: { 'Content-Type': 'application/json' }
            };
            response = await axios.post(`${this.url}/verify-token`, body, config);
        } catch (err) {
            throw new UnauthorizedException('Authentication token validation failed');
        }

        if (response.status == 200 && response.data) {
            request.user = response.data;
            this.addInjectables(request); // if injectable values are configured, add them
            return true;
        } else {
            throw new UnauthorizedException('Authentication token validation failed: no user data');
        }
    }

    private extractTokenFromHeader(request: Request): string | undefined {
        const [type, token] = request.headers.authorization?.split(" ") ?? [];
        return type === "Bearer" && token ? token : undefined;
    }

    private addInjectables(request: any) {
        this.injectUid(request);
        this.injectRole(request);
        this.injectAffiliation(request);
        this.injectRegion(request);
        this.injectNickname(request);
        this.injectDid(request);
    }

    private injectUid(request: any) {
        const injectable = this.configService.get<string>('INJECTED_UID');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "uid"');
            request.user.uid = injectable;
        }
    }

    private injectRole(request: any) {
        const injectable = this.configService.get<string>('INJECTED_ROLE');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "role"');
            request.user.role = injectable;
        }
    }

    private injectAffiliation(request: any) {
        const injectable = this.configService.get<string>('INJECTED_AFFILIATION');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "affiliation"');
            request.user.affiliation = injectable;
        }
    }

    private injectRegion(request: any) {
        const injectable = this.configService.get<string>('INJECTED_REGION');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "region"');
            request.user.region = injectable;
        }
    }

    private injectNickname(request: any) {
        const injectable = this.configService.get<string>('INJECTED_NICKNAME');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "nickname"');
            request.user.nickname = injectable;
        }
    }

    private injectDid(request: any) {
        const injectable = this.configService.get<string>('INJECTED_DID');
        if (injectable && injectable.trim().length > 0) {
            Logger.warn('Injecting FAKE value ' + injectable + ' in security context as "issuerDID"');
            request.user.issuerDID = injectable;
        }
    }
}
