import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import {
    IsAlphanumeric,
    IsEnum,
    IsNotEmpty,
    IsOptional,
    IsString,
    IsUrl,
    Length,
    Matches,
    MaxLength,
    MinLength,
} from 'class-validator';
import { ID_MAXL, TXT_LONG_MAXL, TXT_SHORT_MAXL, ONE_LINER_MAXL, AssetType, SHORTID_MAXL } from '../constants';
import { IsCommaSeparatedAsciiWords, IsCommaSeparatedUrls } from './custom-validators';

/*
    This DTO class supports the DCAT standard for field naming, adding the prefix "dcat_" of "dcterms_" to the field names
    that correspond to the standard, and adding the prefix "fame_" to the field names that are specific to the FAME project.
    Objects of this class can be used as user-provided input in two contexts: publishing a new asset and revising an existing
    asset: annotations are provided for semantics and validation rules in the these contexts, and should be ignored in any
    other context. 
*/
export class AssetDescriptor {

    @ApiPropertyOptional({
        description: 'AID of trusted asset (assigned by system, IGNORED in input)',
        example: 'cx6AcYXZeGgq9euxm',
    })
    @IsOptional()
    @IsString()
    @MinLength(1)
    @MaxLength(ID_MAXL)
    dcterms_identifier?: string;

    @ApiPropertyOptional({
        description: 'PID of publishing organization (assigned by system, IGNORED in input)',
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsOptional()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    dcterms_creator?: string;

    @ApiProperty({
        description: `Title of catalogue entry (REQUIRED for publishing, IGNORED for revision, 1-${ONE_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(ONE_LINER_MAXL)
    @Matches(/^[^\x00-\x1F\x7F<>{}\\]+$/u, {
        message: 'Title contains invalid characters. Please avoid control characters and HTML markup (no newlines allowed).'
    })
    dcterms_title: string;

    @ApiProperty({
        description: `Short description of catalogue entry (REQUIRED for publishing, OPTIONAL for revision, 1-${TXT_SHORT_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(TXT_SHORT_MAXL)
    @Matches(/^[^\x00-\x1F\x7F<>{}\\]*$/u, {
        message: 'Short description contains invalid characters. Please avoid control characters and HTML markup.'
    })
    dcterms_description_short: string;

    @ApiPropertyOptional({
        description: `Long description of catalogue entry with basic Markdown support (OPTIONAL in input, 1-${TXT_LONG_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(TXT_LONG_MAXL)
    @Matches(/^[^\x00-\x08\x0B\x0C\x0E-\x1F\x7F<>\\]*$/u, {
        message: 'Long description contains invalid characters. Use basic Markdown formatting only (**bold**, *italic*, # headings, lists). Avoid links, images, code blocks, and HTML markup.'
    })
    dcterms_description_long?: string;

    @ApiProperty({
        description: `Content type (REQUIRED for publishing, IGNORED for revision, accepted values: [${Object.values(AssetType).join(', ')}])`,
    })
    @IsNotEmpty()
    @IsEnum(AssetType)
    dcterms_type: string;

    @ApiProperty({
        description: 'Content server (REQUIRED for publishing, OPTIONAL for revision, URL string)',
    })
    @IsNotEmpty()
    @IsUrl()
    dcat_endpointURL: string;

    @ApiPropertyOptional({
        description: 'Descriptive/support web page (OPTIONAL in input, URL string) ',
    })
    @IsOptional()
    @IsUrl()
    dcat_landingPage?: string;

    @ApiPropertyOptional({
        description: 'Relevant standards (OPTIONAL in input, comma-separated list of URLs)',
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(TXT_SHORT_MAXL)
    @IsCommaSeparatedUrls() // custom validation, see implementation below
    dcterms_conformsTo?: string;

    @ApiPropertyOptional({
        description: 'Logo of catalogue entry (OPTIONAL in input, URL string)',
    })
    @IsOptional()
    @IsUrl()
    fame_logo?: string;
}
