import { ApiProperty } from '@nestjs/swagger';

export class BlockchainStats {
    
    @ApiProperty({
        description: 'Total number of successful connections established',
        example: 15
    })
    totalConnections: number;

    @ApiProperty({
        description: 'Total number of disconnections (planned and unplanned)',
        example: 2
    })
    totalDisconnections: number;

    @ApiProperty({
        description: 'Total number of reconnection attempts made',
        example: 3
    })
    totalReconnections: number;

    @ApiProperty({
        description: 'Current number of consecutive reconnection attempts',
        example: 0
    })
    currentReconnectAttempts: number;

    @ApiProperty({
        description: 'Maximum allowed reconnection attempts before giving up',
        example: 5
    })
    maxReconnectAttempts: number;

    @ApiProperty({
        description: 'Timestamp of the last successful connection (ISO string)',
        example: '2025-06-09T11:17:53.000Z',
        nullable: true
    })
    lastConnectionTime: string | null;

    @ApiProperty({
        description: 'Timestamp of the last disconnection (ISO string)',
        example: '2025-06-09T10:45:22.000Z',
        nullable: true
    })
    lastDisconnectionTime: string | null;

    @ApiProperty({
        description: 'Current connection status',
        example: true
    })
    isConnected: boolean;

    @ApiProperty({
        description: 'Total uptime in milliseconds since service started',
        example: 8100000
    })
    totalUptimeMs: number;

    @ApiProperty({
        description: 'Average connection duration in milliseconds',
        example: 2730000
    })
    averageConnectionDurationMs: number;

    @ApiProperty({
        description: 'Current connection duration in milliseconds (if connected)',
        example: 1800000,
        nullable: true
    })
    currentConnectionDurationMs: number | null;

    @ApiProperty({
        description: 'Blockchain endpoint URL',
        example: 'ws://blockchain-node:8546'
    })
    endpoint: string;

    @ApiProperty({
        description: 'Chain ID',
        example: '6666'
    })
    chainId: string;

    @ApiProperty({
        description: 'Service start time (ISO string)',
        example: '2025-06-09T09:00:00.000Z'
    })
    serviceStartTime: string;

    @ApiProperty({
        description: 'Last keepalive success timestamp (ISO string)',
        example: '2025-06-09T11:19:52.000Z',
        nullable: true
    })
    lastKeepaliveSuccess: string | null;

    @ApiProperty({
        description: 'Number of failed keepalive attempts',
        example: 0
    })
    failedKeepaliveAttempts: number;

    @ApiProperty({
        description: 'Current block number (from last successful keepalive)',
        example: 11736422,
        nullable: true
    })
    currentBlockNumber: number | null;

    constructor() {
        this.totalConnections = 0;
        this.totalDisconnections = 0;
        this.totalReconnections = 0;
        this.currentReconnectAttempts = 0;
        this.maxReconnectAttempts = 0;
        this.lastConnectionTime = null;
        this.lastDisconnectionTime = null;
        this.isConnected = false;
        this.totalUptimeMs = 0;
        this.averageConnectionDurationMs = 0;
        this.currentConnectionDurationMs = null;
        this.endpoint = '';
        this.chainId = '';
        this.serviceStartTime = new Date().toISOString();
        this.lastKeepaliveSuccess = null;
        this.failedKeepaliveAttempts = 0;
        this.currentBlockNumber = null;
    }

    @ApiProperty({
        description: 'Calculate uptime percentage since service started',
        example: 98.5
    })
    get uptimePercentage(): number {
        const totalTimeMs = Date.now() - new Date(this.serviceStartTime).getTime();
        if (totalTimeMs === 0) return 100;
        return Math.round((this.totalUptimeMs / totalTimeMs) * 100 * 100) / 100; // Round to 2 decimal places
    }

    @ApiProperty({
        description: 'Get human-readable connection status',
        example: 'Connected'
    })
    get connectionStatus(): string {
        if (this.isConnected) {
            return 'Connected';
        } else if (this.currentReconnectAttempts > 0) {
            return `Reconnecting (${this.currentReconnectAttempts}/${this.maxReconnectAttempts})`;
        } else if (this.currentReconnectAttempts >= this.maxReconnectAttempts) {
            return 'Failed - Max reconnection attempts exceeded';
        } else {
            return 'Disconnected';
        }
    }

    @ApiProperty({
        description: 'Get formatted current connection duration',
        example: '30m 15s',
        nullable: true
    })
    get formattedCurrentConnectionDuration(): string | null {
        if (!this.currentConnectionDurationMs) return null;
        return this.formatDuration(this.currentConnectionDurationMs);
    }

    @ApiProperty({
        description: 'Get formatted average connection duration',
        example: '45m 30s'
    })
    get formattedAverageConnectionDuration(): string {
        return this.formatDuration(this.averageConnectionDurationMs);
    }

    @ApiProperty({
        description: 'Get formatted total uptime',
        example: '2h 15m'
    })
    get formattedTotalUptime(): string {
        return this.formatDuration(this.totalUptimeMs);
    }

    private formatDuration(ms: number): string {
        if (ms < 1000) return `${ms}ms`;
        
        const seconds = Math.floor(ms / 1000);
        if (seconds < 60) return `${seconds}s`;
        
        const minutes = Math.floor(seconds / 60);
        if (minutes < 60) return `${minutes}m ${seconds % 60}s`;
        
        const hours = Math.floor(minutes / 60);
        if (hours < 24) return `${hours}h ${minutes % 60}m`;
        
        const days = Math.floor(hours / 24);
        return `${days}d ${hours % 24}h`;
    }
}