import { ApiPropertyOptional } from "@nestjs/swagger";
import { IsInt, IsOptional, Matches, Max, Min, registerDecorator, ValidationArguments, ValidationOptions, ValidatorConstraint, ValidatorConstraintInterface } from "class-validator";

// custom validator: needs to be defined _before_ the DTO class
@ValidatorConstraint({ name: 'OneOfThree', async: false })
export class OneOfThreeConstraint implements ValidatorConstraintInterface {
    validate(_: unknown, args: ValidationArguments) {
        // The "object" here is the entire class instance
        const obj = args.object as any;
        const { duration, retrievals, volume } = obj;

        let count = 0;
        if (duration) count++;
        if (retrievals) count++;
        if (volume) count++;

        return count === 1; // Must be exactly one
    }

    defaultMessage(args: ValidationArguments) {
        return 'Exactly one of [duration, retrievals, volume] must be provided.';
    }
}

// decorator factory used on the _dummyField class property
export function OneOfThree(validationOptions?: ValidationOptions) {
    return function (object: object, propertyName: string) {
        registerDecorator({
            name: 'OneOfThree',
            target: object.constructor,
            propertyName,
            options: validationOptions,
            validator: OneOfThreeConstraint,
        });
    };
}

export class ConsumptionUnit {

    // this dummy property triggers the cross-field check at runtime,
    // the actual validation logic is in OneOfThreeConstraint
    @OneOfThree()
    private _dummyField?: unknown;

    @ApiPropertyOptional({
        description: 'Duration of unrestricted access - accepted values: {nn}m (minutes), {nn}H (hours), {nn}D (days), {nn}M (months), {nn}Y (years)',
    })
    @IsOptional()
    @Matches(/^[0-9]{1,2}[mHDMY]$/, {
        message: 'Duration must be like 30D, 1M, etc.',
    })
    duration?: string;

    @ApiPropertyOptional({
        description: 'Number of content retrievals (e.g., individual downloads or service calls) - integer in the 1-1000000 range',
    })
    @IsOptional()
    @IsInt()
    @Min(1)
    @Max(1000000)
    retrievals?: number;

    @ApiPropertyOptional({
        description: 'Volume of retrieved content (e.g., cumulative size of downloads or served content) - accepted values: {nnnn}B, {nnnn}KB, {nnnn}MB, {n}GB, {nnnn}TB',
    })
    @IsOptional()
    @Matches(/^[0-9]{1,4}(B|KB|MB|GB|TB)$/, {
        message: 'Volume must be like "500MB", "2GB", etc.',
    })
    volume?: string;
}
