import { ValidatorConstraint, ValidatorConstraintInterface, ValidationArguments, isURL, ValidationOptions, registerDecorator } from 'class-validator';
import { PRICE_MAX } from '../constants';

@ValidatorConstraint({ name: 'IsCommaSeparatedUrls', async: false })
export class IsCommaSeparatedUrlsConstraint implements ValidatorConstraintInterface {
    validate(value: any, args?: ValidationArguments): boolean {
        if (typeof value !== 'string') return false;
        const parts = value.split(',');
        return parts.every(part => isURL(part.trim()));
    }

    defaultMessage(args?: ValidationArguments): string {
        return `${args.property}: each comma-separated value must be a valid URL`;
    }
}

export function IsCommaSeparatedUrls(validationOptions?: ValidationOptions) {
    return function (object: any, propertyName: string) {
        registerDecorator({
            name: 'IsCommaSeparatedUrls',
            target: object.constructor,
            propertyName: propertyName,
            options: validationOptions,
            validator: IsCommaSeparatedUrlsConstraint,
        });
    };
}

@ValidatorConstraint({ name: 'IsCommaSeparatedAsciiWords', async: false })
export class IsCommaSeparatedAsciiWordsConstraint
    implements ValidatorConstraintInterface {
    validate(value: any, args?: ValidationArguments): boolean {
        if (typeof value !== 'string') return false;
        const parts = value.split(',').map((p) => p.trim());
        const asciiRegex = /^[ -~]+$/;
        return parts.every((part) => part.length > 0 && asciiRegex.test(part));
    }

    defaultMessage(args?: ValidationArguments): string {
        return `${args.property}: each comma-separated value must contain only printable ASCII characters`;
    }
}

export function IsCommaSeparatedAsciiWords(validationOptions?: ValidationOptions) {
    return function (object: any, propertyName: string) {
        registerDecorator({
            name: 'IsCommaSeparatedAsciiWords',
            target: object.constructor,
            propertyName,
            options: validationOptions,
            validator: IsCommaSeparatedAsciiWordsConstraint,
        });
    };
}

@ValidatorConstraint({ name: 'IsPriceInRange', async: false })
export class IsPriceInRangeConstraint implements ValidatorConstraintInterface {
    validate(value: any, args: ValidationArguments) {
        const num = parseFloat(value);
        return !isNaN(num) && num >= 0 && num <= PRICE_MAX;
    }

    defaultMessage(args: ValidationArguments) {
        return `${args.property}: value must be between 0 and ${PRICE_MAX}`;
    }
}

export function IsPriceInRange(validationOptions?: ValidationOptions) {
    return function (object: any, propertyName: string) {
        registerDecorator({
            target: object.constructor,
            propertyName,
            options: validationOptions,
            constraints: [],
            validator: IsPriceInRangeConstraint,
        });
    };
}
