import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import { PostalAddress } from './postal-address';
import { IsAlphanumeric, IsBIC, IsNotEmpty, IsNumberString, IsOptional, Length, Matches, MaxLength, MinLength, registerDecorator, ValidateNested, ValidationArguments, ValidationOptions, ValidatorConstraint, ValidatorConstraintInterface } from 'class-validator';
import { Type } from 'class-transformer';
import { HALF_LINER_MAXL, TWO_LINER_MAXL } from '../constants';

// custom validator: needs to be defined _before_ the DTO class
@ValidatorConstraint({ name: 'AtLeastOne', async: false })
export class AtLeastOneConstraint implements ValidatorConstraintInterface {
    validate(_: unknown, args: ValidationArguments) {
        // The "object" here is the entire class instance
        const obj = args.object as any;
        const { lei, duns, bic, tin } = obj;

        let count = 0;
        if (lei) count++;
        if (duns) count++;
        if (bic) count++;
        if (tin) count++;

        return count >= 1;
    }

    defaultMessage(args: ValidationArguments) {
        return 'At least one of [lei, duns, bic, tin] must be provided.';
    }
}

// decorator factory used on the _dummyField class property
export function AtLeastOne(validationOptions?: ValidationOptions) {
    return function (object: object, propertyName: string) {
        registerDecorator({
            name: 'AtLeastOne',
            target: object.constructor,
            propertyName,
            options: validationOptions,
            validator: AtLeastOneConstraint,
        });
    };
}

// at least one lei/duns/bic/tin field is REQUIRED for onboarding
export class LegalEntity {

    // this dummy property triggers the cross-field check at runtime,
    // the actual validation logic is in AtLeastOneConstraint
    @AtLeastOne()
    private _dummyField?: unknown;

    @ApiProperty({
        description: `Legal name of the organization (REQUIRED in input, 1-${TWO_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(TWO_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'Legal name must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    legalName: string;

    @ApiProperty({
        description: 'Legal address of the organization (REQUIRED in input)',
    })
    @IsNotEmpty()
    @ValidateNested()
    @Type(() => PostalAddress)
    legalAddress: PostalAddress;

    @ApiPropertyOptional({
        description: 'Legal Entity Identifier (OPTIONAL, 20 alphanumeric characters) - NOTE: at least one of lei, duns, bic, tin is REQUIRED in input',
    })
    @IsOptional()
    @Length(20, 20)
    @IsAlphanumeric()
    lei: string;

    @ApiPropertyOptional({
        description: 'Data Universal Numbering System (OPTIONAL, 9 numeric characters) - NOTE: at least one of lei, duns, bic, tin is REQUIRED in input',
    })
    @IsOptional()
    @Length(9, 9)
    @IsNumberString()
    duns: string;

    @ApiPropertyOptional({
        description: 'Bank Identifier Code, also known as SWIFT code (OPTIONAL, 8 or 11 alphanumeric characters) - NOTE: at least one of lei, duns, bic, tin is REQUIRED in input',
    })
    @IsOptional()
    @IsBIC()
    bic: string;

    @ApiPropertyOptional({
        description: `Taxpayer Identification Number (OPTIONAL, 1-${HALF_LINER_MAXL} characters) - NOTE: at least one of lei, duns, bic, tin is REQUIRED in input`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^(?=.*[A-Za-z0-9])[A-Za-z0-9 ]+$/, {
        message: 'tin must contain only alphanumeric characters and spaces, and cannot consist solely of spaces',
    })
    tin: string;
}
