import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import {
    IsAlphanumeric,
    IsDecimal,
    IsEthereumAddress,
    IsJSON,
    IsNotEmpty,
    IsOptional,
    Matches,
    MaxLength,
    MinLength,
    ValidateNested
} from 'class-validator';
import { ConsumptionUnit } from "./consumption-unit";
import { Type } from 'class-transformer';
import { ID_MAXL, DOCUMENT_MAXL, TXT_LONG_MAXL, TXT_SHORT_MAXL, ONE_LINER_MAXL } from '../constants';
import { IsPriceInRange } from './custom-validators';

/*
    Objects of this class can be used as user-provided input in when publishing a new offering: annotations
    are provided for semantics and validation rules in this context, and should be ignored in any other context. 
*/
export class OfferingDescriptor {

    @ApiProperty({
        description: 'AID of root asset (REQUIRED for publishing, must reference an existing published asset)',
    })
    @MinLength(1)
    @MaxLength(ID_MAXL)
    @IsAlphanumeric()
    asset: string;

    @ApiProperty({
        description: `Offering title (REQUIRED for publishing, 1-${ONE_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(ONE_LINER_MAXL)
    @Matches(/^[^\x00-\x1F\x7F<>{}\\]+$/u, {
        message: 'Offering title contains invalid characters. Please avoid control characters and HTML markup (no newlines allowed).'
    })
    title: string;

    @ApiProperty({
        description: `Offering overview (REQUIRED for publishing, 1-${TXT_LONG_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(TXT_LONG_MAXL)
    @Matches(/^[^\x00-\x1F\x7F<>{}\\]*$/u, {
        message: 'Offering overview contains invalid characters. Please avoid control characters and HTML markup.'
    })
    summary: string;

    @ApiPropertyOptional({
        description: `Limitations in scope with respect to the entire asset (OPTIONAL, 1-${TXT_SHORT_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(TXT_SHORT_MAXL)
    @Matches(/^[^\x00-\x1F\x7F<>{}\\]*$/u, {
        message: 'Limitations in scope contains invalid characters. Please avoid control characters and HTML markup.'
    })
    scope?: string;

    @ApiProperty({
        description: 'Consumption unit specification (REQUIRED for publishing, only one duration/retrievals/volume option is allowed)',
    })
    @IsNotEmpty()
    @ValidateNested()
    @Type(() => ConsumptionUnit)
    unit: ConsumptionUnit;

    @ApiProperty({
        description: 'Price per consumption unit in FDE currency (REQUIRED for publishing, two decimal places)',
    })
    @IsDecimal({ decimal_digits: '2,2', force_decimal: true })
    @IsPriceInRange() // custom validation, see implementation below
    price: string;

    @ApiProperty({
        description: `Formal definition of terms and conditions with basic Markdown support (REQUIRED for publishing, 1-${DOCUMENT_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(DOCUMENT_MAXL)
    @Matches(/^[^\x00-\x08\x0B\x0C\x0E-\x1F\x7F<>\\]*$/u, {
        message: 'Terms and conditions contains invalid characters. Use basic Markdown formatting only (**bold**, *italic*, # headings, lists). Avoid links, images, code blocks, and HTML markup.'
    })
    license: string;

    @ApiPropertyOptional({
        description: `Formal definition of service level with basic Markdown support (OPTIONAL, 1-${DOCUMENT_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(DOCUMENT_MAXL)
    @Matches(/^[^\x00-\x08\x0B\x0C\x0E-\x1F\x7F<>\\]*$/u, {
        message: 'Service level agreement contains invalid characters. Use basic Markdown formatting only (**bold**, *italic*, # headings, lists). Avoid links, images, code blocks, and HTML markup.'
    })
    sla?: string;

    @ApiProperty({
        description: 'TID of beneficiary account (REQUIRED for publishing, must be an enrolled trading account)',
        example: '0x42eA4063fC0D1E9276c54e1C2cC7EE6d5616b67f',
    })
    @IsNotEmpty()
    @IsEthereumAddress()
    tid: string;

    @ApiProperty({
        description: `JSON literal that defines the instructions to the content delivery service (REQUIRED for publishing, 2-${TXT_SHORT_MAXL} characters)`,
    })
    @MinLength(2)
    @MaxLength(TXT_SHORT_MAXL)
    @IsJSON()
    cdsinst: string;
}
