import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import { IsString, IsNotEmpty, MinLength, MaxLength, Matches, IsOptional, IsISO31661Alpha2 } from 'class-validator';
import { HALF_LINER_MAXL, ONE_LINER_MAXL } from '../constants';

export class PostalAddress {

    @ApiProperty({
        description: `First line of address (REQUIRED in input, 1-${ONE_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(ONE_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'First line of address must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    addressLine1: string;

    @ApiPropertyOptional({
        description: `Second line of address (OPTIONAL in input, 1-${ONE_LINER_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(ONE_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'Second line of address must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    addressLine2: string;

    @ApiPropertyOptional({
        description: `Third line of address (OPTIONAL in input, 1-${ONE_LINER_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(ONE_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'Third line of address must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    addressLine3: string;

    @ApiProperty({
        description: `City (REQUIRED in input, 1-${HALF_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'City must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    city: string;

    @ApiPropertyOptional({
        description: `Province or region (OPTIONAL in input, 1-${HALF_LINER_MAXL} characters)`,
    })
    @IsOptional()
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'Province or region must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    region: string;

    @ApiProperty({
        description: `Postal code (REQUIRED in input, 1-${HALF_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$/u, {
        message: 'Postal code must only include Latin letters, numbers, punctuation, and spaces (no newlines).'
    })
    postCode: string;

    @ApiProperty({
        description: `Country code (REQUIRED in input, ISO3166 2-letters identifier)`,
    })
    @IsNotEmpty()
    @IsISO31661Alpha2()
    country: string;
}