import { Injectable, Logger } from "@nestjs/common";
import axios from "axios";
import { ConfigService } from "@nestjs/config";
import { ApmToken } from "../dtos/apm-token";
import { UserInfo } from 'src/auth/user-info';

@Injectable()
export class ApmService {
    private readonly url: any;

    constructor(private readonly configService: ConfigService) {
        this.url = this.configService.get<string>('APM_URL');
    }

    public async createPolicy(aid: string, type: string, user: UserInfo): Promise<void> {
        const assetPolicy = {
            assetId: aid,
            assetType: type,
            accessType: 'ONBOARDED'
        };
        await axios.post(this.url + '/asset-policy-editor', assetPolicy, this.getApmConfig(user));
    }

    public async clonePolicy(oldAid: string, newAid: string, user: UserInfo): Promise<void> {
        const targets = {
            sourceAssetId: oldAid,
            newAssetId: newAid
        };
        await axios.post(this.url + '/asset-policy-editor/clone', targets, this.getApmConfig(user));
    }

    // Returns true is the asset is visible according to APM policies.
    // Note that "visible" means that the asset is registered and that the policy allows the user (as described in the argument) to see it,
    // but it does NOT mean that the asset is active (i.e. it can be archived and still visible).
    // This is the internal equivalent of the APM's open API "asset-visibility/check-one" operation.
    public async checkVisibility(aid: string, user: UserInfo): Promise<boolean> {
        const { data } = await axios.get(this.url + '/asset-visibility/check?assetId=' + aid, this.getApmConfig(user));
        return data.hasVisibility === true;
    }

    // Returns true is the asset is visible according to APM policies, is owned by the user (as described in the argument) and is active.
    // This is similar to the APM's open API "asset-access/check-one" operation, except that APM's operation does not check for the active status.
    public async checkEditable(aid: string, user: UserInfo): Promise<boolean> {
        try {
            const { data } = await axios.get(this.url + '/asset-policy-editor/retrieve?assetId=' + aid, this.getApmConfig(user));
            return data.active === true;
        } catch (err) {
            return false;
        }
    }

    // Returns the status string of the asset in APM. If the asset is not visible to the user (as described in the argument), an error is thrown.
    public async getStatus(aid: string, user: UserInfo): Promise<string> {
        const { data } = await axios.get(this.url + '/asset-policy-editor/retrieve?assetId=' + aid, this.getApmConfig(user));
        return data;
    }

    public async setArchived(aid: string, user: UserInfo): Promise<void> {
        await axios.put(this.url + '/asset-policy-editor/archive?assetId=' + aid, null, this.getApmConfig(user));
    }

    private getApmConfig(user: UserInfo): any {
        const attributes = {
            country: user.region,
            role: user.role,
            organizationName: 'N/A',
            organizationType: 'N/A'
        };
        const token = new ApmToken(user.uid, user.affiliation, attributes);
        const jsonXHeader = JSON.stringify(token);
        const base64XHeader = Buffer.from(jsonXHeader).toString('base64');
        return {
            headers: {
                'X-Identity': base64XHeader,
                'Content-Type': 'application/json',
            }
        };
    }
}
