import { Injectable, Logger, OnModuleDestroy } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { getPrivateKeyPermission, getPrivateKeyPT } from 'src/utils/generic-utils';
import Web3 from 'web3';
import * as fs from 'fs';
import * as path from 'path';
import { BlockchainStats } from 'src/dtos/blockchain-stats';

@Injectable()
export class BlockchainService implements OnModuleDestroy {
    private blockchain: any;
    private utils: any;
    private chainid: string;
    private tracingAddress: string;
    private tracingAbi: any;
    private tracingEventsAbi: any;
    private provenanceAddress: string;
    private provenanceAbi: any;
    private permissionAddress: string;
    private permissionAbi: any;
    private transactionQueues: { [address: string]: Promise<any> } = {};

    public tracingContract: any;
    public provenanceContract: any;
    public permissionContract: any;

    // Connection management properties
    private provider: any;
    private keepaliveInterval: NodeJS.Timeout | null = null;
    private endpoint: string;
    private keepaliveIntervalMs: number;
    private isConnected: boolean = false;
    private reconnectAttempts: number = 0;
    private maxReconnectAttempts: number = 5;
    private reconnectDelay: number = 1000;
    private reconnectTimeout: NodeJS.Timeout | null = null;

    // Connection statistics tracking
    private connectionStats: {
        totalConnections: number;
        totalDisconnections: number;
        totalReconnections: number;
        lastConnectionTime: Date | null;
        lastDisconnectionTime: Date | null;
        serviceStartTime: Date;
        connectionDurations: number[];
        totalUptimeMs: number;
        lastKeepaliveSuccess: Date | null;
        failedKeepaliveAttempts: number;
        currentBlockNumber: number | null;
    };

    constructor(private config: ConfigService, private logger: Logger) {
        this.endpoint = config.get<string>('CHAIN_URL');
        this.keepaliveIntervalMs = config.get<number>('CHAIN_KEEPALIVE', 30000);
        this.chainid = config.get<string>('CHAIN_ID');

        // Initialize connection statistics
        this.connectionStats = {
            totalConnections: 0,
            totalDisconnections: 0,
            totalReconnections: 0,
            lastConnectionTime: null,
            lastDisconnectionTime: null,
            serviceStartTime: new Date(),
            connectionDurations: [],
            totalUptimeMs: 0,
            lastKeepaliveSuccess: null,
            failedKeepaliveAttempts: 0,
            currentBlockNumber: null,
        };

        // Initialize connection
        this.initializeConnection();

        // Initialize contracts and other components
        this.initializeContracts();

        // Start uptime tracking
        this.startUptimeTracking();
    }

    /**
     * Get current connection statistics
     */
    public getStats(): BlockchainStats {
        const stats = new BlockchainStats();

        stats.totalConnections = this.connectionStats.totalConnections;
        stats.totalDisconnections = this.connectionStats.totalDisconnections;
        stats.totalReconnections = this.connectionStats.totalReconnections;
        stats.currentReconnectAttempts = this.reconnectAttempts;
        stats.maxReconnectAttempts = this.maxReconnectAttempts;
        stats.lastConnectionTime = this.connectionStats.lastConnectionTime?.toISOString() || null;
        stats.lastDisconnectionTime = this.connectionStats.lastDisconnectionTime?.toISOString() || null;
        stats.isConnected = this.isConnected;
        stats.totalUptimeMs = this.connectionStats.totalUptimeMs;
        stats.averageConnectionDurationMs = this.calculateAverageConnectionDuration();
        stats.currentConnectionDurationMs = this.getCurrentConnectionDuration();
        stats.endpoint = this.endpoint;
        stats.chainId = this.chainid;
        stats.serviceStartTime = this.connectionStats.serviceStartTime.toISOString();
        stats.lastKeepaliveSuccess = this.connectionStats.lastKeepaliveSuccess?.toISOString() || null;
        stats.failedKeepaliveAttempts = this.connectionStats.failedKeepaliveAttempts;
        stats.currentBlockNumber = this.connectionStats.currentBlockNumber;

        return stats;
    }

    private startUptimeTracking(): void {
        // Update uptime every second
        setInterval(() => {
            if (this.isConnected) {
                this.connectionStats.totalUptimeMs += 1000;
            }
        }, 1000);
    }

    private calculateAverageConnectionDuration(): number {
        if (this.connectionStats.connectionDurations.length === 0) return 0;

        const total = this.connectionStats.connectionDurations.reduce((sum, duration) => sum + duration, 0);
        return Math.round(total / this.connectionStats.connectionDurations.length);
    }

    private getCurrentConnectionDuration(): number | null {
        if (!this.isConnected || !this.connectionStats.lastConnectionTime) return null;

        return Date.now() - this.connectionStats.lastConnectionTime.getTime();
    }

    private recordConnection(): void {
        this.connectionStats.totalConnections++;
        this.connectionStats.lastConnectionTime = new Date();
        this.logger.debug(`Connection recorded. Total connections: ${this.connectionStats.totalConnections}`);
    }

    private recordDisconnection(): void {
        const now = new Date();
        this.connectionStats.totalDisconnections++;

        // Record connection duration if we have a connection start time
        if (this.connectionStats.lastConnectionTime) {
            const duration = now.getTime() - this.connectionStats.lastConnectionTime.getTime();
            this.connectionStats.connectionDurations.push(duration);

            // Keep only the last 100 durations to prevent memory growth
            if (this.connectionStats.connectionDurations.length > 100) {
                this.connectionStats.connectionDurations.shift();
            }
        }

        this.connectionStats.lastDisconnectionTime = now;
        this.logger.debug(`Disconnection recorded. Total disconnections: ${this.connectionStats.totalDisconnections}`);
    }

    private recordReconnectionAttempt(): void {
        this.connectionStats.totalReconnections++;
        this.logger.debug(`Reconnection attempt recorded. Total reconnections: ${this.connectionStats.totalReconnections}`);
    }

    // Public method to check connection health
    public isConnectionHealthy(): boolean {
        return this.isConnected && this.provider && this.provider.connected !== false;
    }

    // Public method to force reconnection
    public async forceReconnect(): Promise<void> {
        this.logger.warn('Forcing blockchain reconnection');
        this.reconnectAttempts = 0;
        this.initializeConnection();
    }

    private initializeConnection(): void {
        try {
            // Clean up existing connection if any
            this.cleanupConnection();

            // Create new WebSocket provider
            this.provider = new Web3.providers.WebsocketProvider(this.endpoint);

            // Set up event handlers
            this.setupProviderEventHandlers();

            // Initialize Web3 instance
            const web3 = new Web3(this.provider);
            this.blockchain = web3.eth;
            this.utils = web3.utils;

            // Initialize wallet accounts
            this.initializeAccounts();

            // Start keepalive mechanism
            this.startKeepalive();

            this.logger.log('Blockchain connection initialized successfully');

        } catch (error) {
            this.logger.error('Failed to initialize blockchain connection:', error);
            this.scheduleReconnect();
        }
    }

    private setupProviderEventHandlers(): void {
        this.provider.on('connect', () => {
            this.logger.log('WebSocket connected to blockchain');
            this.isConnected = true;
            this.reconnectAttempts = 0;
            this.recordConnection();
        });

        this.provider.on('disconnect', (error: any) => {
            this.logger.warn('WebSocket disconnected from blockchain:', error);
            if (this.isConnected) {
                this.recordDisconnection();
            }
            this.isConnected = false;
            this.scheduleReconnect();
        });

        this.provider.on('error', (error: any) => {
            this.logger.error('WebSocket error:', error);
            if (this.isConnected) {
                this.recordDisconnection();
            }
            this.isConnected = false;
            this.scheduleReconnect();
        });

        this.provider.on('end', () => {
            this.logger.warn('WebSocket connection ended');
            if (this.isConnected) {
                this.recordDisconnection();
            }
            this.isConnected = false;
            this.scheduleReconnect();
        });
    }

    private initializeAccounts(): void {
        // Clear existing wallet
        this.blockchain.accounts.wallet.clear();

        // Add accounts
        this.blockchain.accounts.wallet.add(getPrivateKeyPT(this.config));
        this.blockchain.accounts.wallet.add(getPrivateKeyPermission(this.config));

        this.logger.debug(`Provenance and Tracing blockchain account ${this.blockchain.accounts.wallet[0].address} initialized`);
        this.logger.debug(`Permission blockchain account ${this.blockchain.accounts.wallet[1].address} initialized`);
        this.logger.debug(`Chain ID is ${this.chainid}`);
    }

    private initializeContracts(): void {
        // init provenance contract
        this.provenanceAddress = this.config.get<string>('CONTRACT_PROVENANCE_ADDR');
        let abiPath = this.config.get<string>('CONTRACT_PROVENANCE_ABI');
        let abiAbsPath = path.join(__dirname, '..', '..', 'src', abiPath);
        this.provenanceAbi = JSON.parse(fs.readFileSync(abiAbsPath, 'utf8'));
        this.provenanceContract = new this.blockchain.Contract(this.provenanceAbi, this.provenanceAddress);
        this.logger.debug(`Provenance contract ${this.provenanceAddress} initialized`);

        // init tracing contract
        this.tracingAddress = this.config.get<string>('CONTRACT_TRACING_ADDR');
        abiPath = this.config.get<string>('CONTRACT_TRACING_ABI');
        abiAbsPath = path.join(__dirname, '..', '..', 'src', abiPath);
        this.tracingAbi = JSON.parse(fs.readFileSync(abiAbsPath, 'utf8'));
        this.tracingEventsAbi = this.tracingAbi.find((abi: any) => abi.name === 'AssetEvent').inputs;
        this.tracingContract = new this.blockchain.Contract(this.tracingAbi, this.tracingAddress);
        this.logger.debug(`Tracing contract ${this.tracingAddress} initialized`);

        // init permission contract
        this.permissionAddress = this.config.get<string>('CONTRACT_PERMISSION_ADDR');
        abiPath = this.config.get<string>('CONTRACT_PERMISSION_ABI');
        abiAbsPath = path.join(__dirname, '..', '..', 'src', abiPath);
        this.permissionAbi = JSON.parse(fs.readFileSync(abiAbsPath, 'utf8'));
        this.permissionContract = new this.blockchain.Contract(this.permissionAbi, this.permissionAddress);
        this.logger.debug(`Permission contract ${this.permissionAddress} initialized`);
    }

    private startKeepalive(): void {
        // Clear existing interval
        if (this.keepaliveInterval) {
            clearInterval(this.keepaliveInterval);
        }

        this.keepaliveInterval = setInterval(async () => {
            try {
                if (!this.isConnected) {
                    this.logger.debug('Skipping keepalive - not connected');
                    return;
                }

                const blockNumber = await this.blockchain.getBlockNumber();
                this.connectionStats.lastKeepaliveSuccess = new Date();
                // Convert BigInt to number for JSON serialization
                this.connectionStats.currentBlockNumber = typeof blockNumber === 'bigint'
                    ? Number(blockNumber)
                    : blockNumber;
                this.connectionStats.failedKeepaliveAttempts = 0; // Reset on success
            } catch (err) {
                this.connectionStats.failedKeepaliveAttempts++;
                this.logger.error('Keepalive request failed:', err);
                this.isConnected = false;
                this.scheduleReconnect();
            }
        }, this.keepaliveIntervalMs);

        this.logger.debug(`Blockchain connection keepalive cycle of ${this.keepaliveIntervalMs}ms started`);
    }

    private scheduleReconnect(): void {
        // Prevent multiple reconnection attempts
        if (this.reconnectTimeout) {
            return;
        }

        // Don't reconnect if we've exceeded max attempts
        if (this.reconnectAttempts >= this.maxReconnectAttempts) {
            this.logger.error(`Max reconnection attempts (${this.maxReconnectAttempts}) exceeded. Manual intervention required.`);
            return;
        }

        this.reconnectAttempts++;
        this.recordReconnectionAttempt();
        const delay = this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1);

        this.logger.warn(`Scheduling reconnection attempt ${this.reconnectAttempts}/${this.maxReconnectAttempts} in ${delay}ms`);

        this.reconnectTimeout = setTimeout(() => {
            this.reconnectTimeout = null;
            this.logger.log(`Attempting to reconnect (attempt ${this.reconnectAttempts}/${this.maxReconnectAttempts})`);
            this.initializeConnection();
        }, delay);
    }

    private cleanupConnection(): void {
        // Clear keepalive interval
        if (this.keepaliveInterval) {
            clearInterval(this.keepaliveInterval);
            this.keepaliveInterval = null;
        }

        // Clear reconnect timeout
        if (this.reconnectTimeout) {
            clearTimeout(this.reconnectTimeout);
            this.reconnectTimeout = null;
        }

        // Record disconnection if we were connected
        if (this.isConnected) {
            this.recordDisconnection();
        }

        // Close existing provider connection
        if (this.provider) {
            try {
                // Remove all listeners to prevent memory leaks
                this.provider.removeAllListeners();

                // Disconnect the provider
                if (typeof this.provider.disconnect === 'function') {
                    this.provider.disconnect();
                } else if (typeof this.provider.connection?.close === 'function') {
                    this.provider.connection.close();
                }
            } catch (error) {
                this.logger.warn('Error during provider cleanup:', error);
            }
            this.provider = null;
        }

        this.isConnected = false;
    }

    // Implement OnModuleDestroy for proper cleanup when app shuts down
    async onModuleDestroy(): Promise<void> {
        this.logger.log('Cleaning up blockchain service...');
        this.cleanupConnection();
    }

    public executeTransaction(contract: any, input: string, onSuccess: () => void, onError: () => void) {
        this.executeTransactionWithRetry(contract, input, onSuccess, onError, 3);
    }

    private async executeTransactionWithRetry(
        contract: any, 
        input: string, 
        onSuccess: () => void, 
        onError: () => void, 
        maxRetries: number,
        attempt: number = 1
    ) {
        // Check connection health before executing
        if (!this.isConnectionHealthy()) {
            this.logger.error('Cannot execute transaction - blockchain connection is not healthy');
            try {
                onError();
            } catch (err) {
                this.logger.error(`Error executing transaction "error" callback: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
            return;
        }

        // use the account that matches the contract
        const account = (contract === this.permissionContract)
            ? this.blockchain.accounts.wallet[1]
            : this.blockchain.accounts.wallet[0];

        try {
            const trx = await this.getSignedTransaction(contract, input, account);

            this.blockchain.sendSignedTransaction(trx)
                .on('transactionHash', (hash: string) => {
                    this.logger.log(`Transaction hash: ${hash}`);
                    // Store the transaction hash in a property that can be accessed by the callback
                    (onSuccess as any).__txHash = hash;
                })
                .on('receipt', (receipt: any) => {
                    this.logger.debug(`Transaction receipt received - Block: ${receipt.blockNumber}, Gas used: ${receipt.gasUsed}`);
                    // Pass receipt to success callback
                    (onSuccess as any).__receipt = receipt;
                    try {
                        onSuccess();
                    } catch (err) {
                        this.logger.error(`Error executing transaction "success" callback: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                    }
                })
                .on('error', (err: any) => {
                    this.handleTransactionError(err, contract, input, onSuccess, onError, maxRetries, attempt, account);
                })
                .catch((err: any) => {
                    this.handleTransactionError(err, contract, input, onSuccess, onError, maxRetries, attempt, account);
                });
        } catch (err) {
            this.handleTransactionError(err, contract, input, onSuccess, onError, maxRetries, attempt, account);
        }
    }

    private async handleTransactionError(
        err: any,
        contract: any,
        input: string,
        onSuccess: () => void,
        onError: () => void,
        maxRetries: number,
        attempt: number,
        account: any
    ) {
        this.logger.error(`Error executing blockchain transaction (attempt ${attempt}/${maxRetries}): ${err.name}, Message: ${err.message}`);

        // If it's a nonce conflict and we have retries left, retry with fresh nonce
        if (this.isNonceError(err) && attempt < maxRetries) {
            this.logger.warn(`Nonce conflict detected for ${account.address}, retrying with fresh nonce (attempt ${attempt + 1}/${maxRetries})`);
            
            // Exponential backoff: 200ms, 400ms, 800ms
            const delay = 200 * Math.pow(2, attempt - 1);
            await this.delay(delay);
            
            // Retry the entire transaction
            this.executeTransactionWithRetry(contract, input, onSuccess, onError, maxRetries, attempt + 1);
            return;
        }

        // If it's a connection error, trigger reconnection
        if (this.isConnectionError(err)) {
            this.isConnected = false;
            this.scheduleReconnect();
        }

        // Final failure - call error callback
        try {
            onError();
        } catch (callbackErr) {
            this.logger.error(`Error executing transaction "error" callback: ${callbackErr.name}, Message: ${callbackErr.message}, Stack: ${callbackErr.stack}`);
        }
    }

    private isConnectionError(error: any): boolean {
        const connectionErrors = [
            'connection not open',
            'websocket connection closed',
            'connection closed',
            'network error',
            'timeout',
            'econnreset',
            'econnrefused'
        ];

        const errorMessage = error?.message?.toLowerCase() || '';
        return connectionErrors.some(pattern => errorMessage.includes(pattern));
    }

    private isNonceError(error: any): boolean {
        const nonceErrors = [
            'nonce too low',
            'nonce too high',
            'transaction nonce is too low',
            'known transaction'
        ];

        const errorMessage = error?.message?.toLowerCase() || '';
        return nonceErrors.some(pattern => errorMessage.includes(pattern));
    }

    public annotateTracingTrxInput(obj: any) {
        obj.addr = this.tracingAddress;
        obj.abi = JSON.stringify(this.tracingAbi);
        return obj;
    }

    public async getTracingEventsSubscription(): Promise<any> {
        // Wait for connection to be ready before creating subscription
        await this.waitForConnection();

        const eventSignatureHash = this.utils.sha3('AssetEvent(string,string)');
        return await this.blockchain.subscribe('logs', {
            address: this.tracingAddress,
            topics: [eventSignatureHash],
        });
    }

    // Helper method to wait for connection to be ready
    private async waitForConnection(maxWaitMs: number = 10000): Promise<void> {
        const startTime = Date.now();

        while (!this.isConnectionHealthy() && (Date.now() - startTime) < maxWaitMs) {
            this.logger.debug('Waiting for blockchain connection to be ready...');
            await new Promise(resolve => setTimeout(resolve, 100));
        }

        if (!this.isConnectionHealthy()) {
            throw new Error('Cannot create subscription - blockchain connection is not healthy after waiting');
        }
    }

    public decodeTracingLogs(log: any): any {
        return this.blockchain.abi.decodeLog(
            this.tracingEventsAbi,
            log.data,
            log.topics.slice(1)
        );
    }

    public getCanonicalHash(input: string): any {
        return this.utils.keccak256(input);
    }

    public async getTransactionReceipt(txHash: string): Promise<any> {
        return await this.blockchain.getTransactionReceipt(txHash);
    }

    public getTracingContractAddress(): string {
        return this.tracingAddress;
    }

    private delay(ms: number): Promise<void> {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    private async getAndIncrementNonce(account: any, maxRetries: number = 3): Promise<number> {
        const address = account.address;
        
        // Serialize transactions for this address
        if (this.transactionQueues[address]) {
            await this.transactionQueues[address];
        }
        
        this.transactionQueues[address] = this.performFreshNonceRetry(address, maxRetries);
        
        try {
            return await this.transactionQueues[address];
        } finally {
            delete this.transactionQueues[address];
        }
    }

    private async performFreshNonceRetry(address: string, maxRetries: number): Promise<number> {
        for (let attempt = 1; attempt <= maxRetries; attempt++) {
            try {
                // Always get fresh nonce from blockchain
                const nonce = await this.blockchain.getTransactionCount(address, 'pending');
                this.logger.debug(`Using fresh nonce ${nonce} for address ${address} (attempt ${attempt}/${maxRetries})`);
                return nonce;
            } catch (error) {
                this.logger.warn(`Failed to get nonce for ${address} (attempt ${attempt}/${maxRetries}): ${error.message}`);
                
                if (attempt === maxRetries) {
                    throw new Error(`Failed to get nonce after ${maxRetries} attempts: ${error.message}`);
                }
                
                // Exponential backoff: 100ms, 200ms, 400ms
                const delay = 100 * Math.pow(2, attempt - 1);
                this.logger.debug(`Retrying nonce fetch in ${delay}ms`);
                await this.delay(delay);
            }
        }
    }

    private async getSignedTransaction(contract: any, input: string, account: any): Promise<string> {
        const nonce = await this.getAndIncrementNonce(account);

        // try to estimate gas, fallback if necessary
        let gasLimit: number;
        try {
            gasLimit = await this.blockchain.estimateGas({
                from: account.address,
                to: contract.options.address,
                data: input,
            });
        } catch (error) {
            gasLimit = 3000000;
        }

        const transaction = {
            from: account.address,
            nonce: this.utils.toHex(nonce),
            gasPrice: this.utils.toHex(0),
            gasLimit: this.utils.toHex(gasLimit),
            to: contract.options.address,
            data: input,
            chainId: this.chainid,
        };

        const signedTx = await this.blockchain.accounts.signTransaction(transaction, account.privateKey);
        return signedTx.rawTransaction;
    }
}