import { Injectable, Logger } from "@nestjs/common";
import axios, { AxiosInstance, AxiosResponse } from "axios";
import { ConfigService } from "@nestjs/config";
import { UserRequest } from "../dtos/user-request";
import { ErrorCondition, throwExceptionWithErrorCondition } from 'src/utils/generic-utils';
import { Account } from 'src/dtos/account';
import { plainToInstance } from 'class-transformer';

@Injectable()
export class GovService {

    private readonly proxy: AxiosInstance = null;

    constructor(private config: ConfigService) {
        this.proxy = axios.create({
            baseURL: this.config.get<string>('GOV_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
    }


    public enqueueRequest(rid: string, requestor: string, payload: string, message: string) {
        const request = new UserRequest();
        request.rid = rid;
        request.requestor = requestor;
        request.payload = payload;
        request.message = message;

        this.proxy.post('/rqueue', request)
            .then(() => {
                Logger.debug('User request ' + request.rid + ' has been enqueued');
            })
            .catch((error) => {
                Logger.warn('Failed to enqueue user request with id ' + request.rid, error.stack);
            });
    }

    public confirmRequest(rid: string, message: string) {
        const config = {
            params: {
                rid,
                status: 'PROCESSED',
                message
            }
        };
        this.proxy.put(`/rqueue/${rid}`, {}, config)
            .then(() => {
                Logger.debug('User request ' + rid + ' has been set as PROCESSED');
            })
            .catch((error) => {
                Logger.warn('Failed to update user request with id ' + rid, error.stack);
            });
    }

    public rejectRequest(rid: string, message: string) {
        const config = {
            params: {
                rid,
                status: 'FAILED',
                message
            }
        };
        this.proxy.put(`/rqueue/${rid}`, {}, config)
            .then(() => {
                Logger.debug('User request ' + rid + ' has been set as FAILED');
            })
            .catch((error) => {
                Logger.warn('Failed to update user request with id ' + rid, error.stack);
            });
    }

    public async checkAccountAffiliation(tid: string, pid: string): Promise<boolean> {
        let response: AxiosResponse;
        try {
            response = await this.proxy.get('/accounts/' + tid);
        } catch (err) {
            if (err.response && err.response.status === 404) {
                return false;
            } else {
                Logger.error('Error calling GOV account lookup service', err);
                throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Could not communicate with the GOV account lookup service');
            }
        }

        let account: Account = null;
        try {
            account = plainToInstance(Account, response.data);
        } catch (err) {
            Logger.error('Error parsing response from GOV account lookup service', err);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Unexpected response from the GOV account lookup service');
        }

        // account must be active and owned by the given PID
        return (account.owningMember === pid && !account.disenrolled);
    }
}
