import { NestFactory } from '@nestjs/core';
import { DocumentBuilder, SwaggerModule } from '@nestjs/swagger';
import { AppModule } from './app.module';
import { Logger, ValidationPipe } from "@nestjs/common";
import * as fs from "fs";
import * as path from "path";

async function bootstrap() {

    const ssl = process.env.SSL === 'true' || false;
    let httpsOptions = null;
    if (ssl) {
        const keyPath = process.env.SSL_KEY_PATH || '';
        const certPath = process.env.SSL_CERT_PATH || '';
        httpsOptions = {
            key: fs.readFileSync(path.join(__dirname, keyPath)),
            cert: fs.readFileSync(path.join(__dirname, certPath)),
        };
    }
    const app = await NestFactory.create(AppModule, { httpsOptions });

    // enable strict validation of input data through decorators on DTO classes
    app.useGlobalPipes(
        new ValidationPipe({
            whitelist: true,
            forbidNonWhitelisted: true,
            skipMissingProperties: false,
            transform: true,
        }),
    );

    app.enableCors();

    const config = new DocumentBuilder()
        .setTitle('FAME Provenance & Tracing API')
        .setDescription('')
        .setVersion('1.0')
        .build();
    const document = SwaggerModule.createDocument(app, config);

    let fileName = 'swagger-specification.';
    fileName += process.env.INTERNAL_API.trim() === 'true' ? 'private' : 'public';
    fileName += '.json';
    fs.writeFileSync(fileName, JSON.stringify(document));

    SwaggerModule.setup('swagger', app, document);

    const port = process.env.INTERNAL_API.trim() === 'true' ? 7010 : 7005;
    const hostname = process.env.HOSTNAME || 'localhost';
    const address = 'http' + (ssl ? 's' : '') + '://' + hostname + ':' + port;
    await app.listen(port, hostname, () => {
        Logger.log('Listening at ' + address);
        Logger.log('Swagger is running at ' + address + '/swagger');
    });
}
bootstrap();
