#!/usr/bin/env node

import { ConfigService } from '@nestjs/config';
import { BlockchainService } from '../integration/blockchain-service';
import { FdacService } from '../integration/fdac-service';
import { Logger } from '@nestjs/common';
import * as fs from 'fs';
import * as path from 'path';

interface SourceRecord {
    assetId: string;
    value: {
        pid?: string;
        [key: string]: any;
    };
}

interface MigrationRecord {
    aid: string;
    hash: string;
    tid: string;
    registered: number;
    predecessor: string;
    successor: string;
}

class MigrationUtility {
    private blockchain: BlockchainService;
    private fdac: FdacService;
    private config: ConfigService;
    private logger: Logger;
    private sourceData: SourceRecord[] = [];
    private migrationData: MigrationRecord[] = [];
    private currentIndex: number = 0;
    private totalRecords: number = 0;
    private subscription: any = null;
    private migrationAccount: string;
    private transactionTimeout: NodeJS.Timeout | null = null;
    private readonly TRANSACTION_TIMEOUT_MS = 30000; // 30 seconds timeout per transaction (fast private blockchain)
    private currentTransactionHash: string | null = null;

    constructor() {
        this.logger = new Logger('MigrationUtility');
        this.initializeServices();
    }

    private initializeServices(): void {
        // Initialize configuration service
        this.config = new ConfigService();

        // Initialize blockchain service
        this.blockchain = new BlockchainService(this.config, this.logger);

        // Initialize FDAC service
        this.fdac = new FdacService(this.config);

        // Get migration account from environment
        this.migrationAccount = this.config.get<string>('MIGRATION_ACCOUNT');
        if (!this.migrationAccount) {
            throw new Error('MIGRATION_ACCOUNT environment variable is required');
        }

        this.logger.log('Migration utility initialized');
        this.logger.log(`Using migration account: ${this.migrationAccount}`);
    }

    public async loadMigrationData(filePath: string): Promise<void> {
        try {
            if (!fs.existsSync(filePath)) {
                throw new Error(`Migration file not found: ${filePath}`);
            }

            const fileContent = fs.readFileSync(filePath, 'utf8');
            const rawData = JSON.parse(fileContent);

            if (!Array.isArray(rawData)) {
                throw new Error('Migration file must contain an array of records');
            }

            this.sourceData = this.validateSourceData(rawData);
            const filteredData = this.filterRecordsWithPid(this.sourceData);

            this.logger.log(`Loaded ${rawData.length} total records, ${filteredData.length} records have 'pid' property`);

            // Process filtered records to create migration data
            await this.processMigrationRecords(filteredData);

        } catch (error) {
            this.logger.error(`Failed to load migration data: ${error.message}`);
            throw error;
        }
    }

    private validateSourceData(data: any[]): SourceRecord[] {
        return data.map((record, index) => {
            if (!record.assetId || typeof record.assetId !== 'string') {
                throw new Error(`Record ${index}: 'assetId' is required and must be a string`);
            }
            if (!record.value || typeof record.value !== 'object') {
                throw new Error(`Record ${index}: 'value' is required and must be an object`);
            }

            return {
                assetId: record.assetId,
                value: record.value
            };
        });
    }

    private filterRecordsWithPid(data: SourceRecord[]): SourceRecord[] {
        return data.filter(record =>
            record.value &&
            typeof record.value.pid === 'string' &&
            record.value.pid.trim().length > 0
        );
    }

    private async processMigrationRecords(filteredData: SourceRecord[]): Promise<void> {
        this.migrationData = [];
        const currentTimestamp = Math.floor(Date.now() / 1000); // Current timestamp in seconds

        for (let i = 0; i < filteredData.length; i++) {
            const record = filteredData[i];
            try {
                this.logger.log(`Processing asset ${i + 1}/${filteredData.length}: ${record.assetId}`);

                // Get canonical description from FDAC
                const canonicalDescription = await this.fdac.getAssetCanonicalDescription(record.assetId);

                if (!canonicalDescription) {
                    this.logger.warn(`Skipping asset ${record.assetId}: not found in FDAC`);
                    continue;
                }

                // Calculate hash using the same method as AssetService line 312
                const hash = this.blockchain.getCanonicalHash(JSON.stringify(canonicalDescription));

                // Create migration record
                const migrationRecord: MigrationRecord = {
                    aid: record.assetId,
                    hash: hash,
                    tid: this.migrationAccount,
                    registered: currentTimestamp,
                    predecessor: '',
                    successor: ''
                };

                this.migrationData.push(migrationRecord);
                this.logger.log(`Prepared migration record for ${record.assetId} with hash ${hash}`);

            } catch (error) {
                this.logger.error(`Error processing asset ${record.assetId}: ${error.message}`);
                throw new Error(`Failed to process asset ${record.assetId}: ${error.message}`);
            }
        }

        this.totalRecords = this.migrationData.length;
        this.currentIndex = 0;

        this.logger.log(`Successfully prepared ${this.totalRecords} migration records`);
    }

    public async startMigration(): Promise<void> {
        if (this.migrationData.length === 0) {
            throw new Error('No migration data loaded. Call loadMigrationData() first.');
        }

        this.logger.log(`Starting migration of ${this.totalRecords} records...`);

        // Wait for blockchain connection to stabilize (max 30 seconds)
        this.logger.log('Waiting for blockchain connection to stabilize...');
        const connectionReady = await this.waitForStableConnection(30000);

        if (!connectionReady) {
            throw new Error('Blockchain connection did not stabilize within 30 seconds. Please check your CHAIN_URL and ensure the blockchain node is accessible.');
        }

        this.logger.log('Blockchain connection is stable. Proceeding with migration...');

        // Subscribe to blockchain events before starting migration
        await this.subscribeToContractEvents();

        // Start processing the first record
        await this.processNextRecord();
    }

    private async waitForStableConnection(maxWaitMs: number): Promise<boolean> {
        const startTime = Date.now();
        let consecutiveHealthyChecks = 0;
        const requiredConsecutiveChecks = 3; // Need 3 consecutive healthy checks

        while ((Date.now() - startTime) < maxWaitMs) {
            if (this.blockchain.isConnectionHealthy()) {
                consecutiveHealthyChecks++;
                if (consecutiveHealthyChecks >= requiredConsecutiveChecks) {
                    return true;
                }
            } else {
                consecutiveHealthyChecks = 0;
            }
            await new Promise(resolve => setTimeout(resolve, 500));
        }

        return false;
    }

    private async verifyTransactionSuccess(txHash: string, expectedAid: string): Promise<void> {
        try {
            this.logger.debug(`Verifying transaction ${txHash} for AID: ${expectedAid}`);

            // Get the transaction receipt
            const receipt = await this.blockchain.getTransactionReceipt(txHash);

            if (!receipt) {
                this.logger.warn(`No receipt found for transaction ${txHash}, will wait for event`);
                return;
            }

            // Log full receipt status for debugging
            this.logger.debug(`Receipt status: ${receipt.status} (type: ${typeof receipt.status})`);
            this.logger.debug(`Receipt gasUsed: ${receipt.gasUsed}`);
            this.logger.debug(`Receipt from: ${receipt.from}`);
            this.logger.debug(`Receipt to: ${receipt.to}`);

            // Check if transaction succeeded
            if (receipt.status === false || receipt.status === '0x0' || receipt.status === 0) {
                this.logger.error(`Transaction ${txHash} REVERTED on-chain (status: ${receipt.status})`);
                if (this.transactionTimeout) {
                    clearTimeout(this.transactionTimeout);
                    this.transactionTimeout = null;
                }
                this.handleMigrationFailure(`Transaction reverted on-chain for AID: ${expectedAid}`);
                return;
            }

            // Parse logs to find MIGRATED event
            this.logger.debug(`Transaction succeeded (status: ${receipt.status}), checking logs for MIGRATED event...`);
            this.logger.debug(`Receipt has ${receipt.logs?.length || 0} log entries`);

            const contractAddress = this.blockchain.getTracingContractAddress();
            this.logger.debug(`Looking for logs from contract: ${contractAddress}`);

            let foundMigratedEvent = false;

            if (receipt.logs && receipt.logs.length > 0) {
                for (let i = 0; i < receipt.logs.length; i++) {
                    const log = receipt.logs[i];
                    this.logger.debug(`Log ${i}: address=${log.address}, topics=${log.topics?.length || 0}`);

                    try {
                        if (log.address.toLowerCase() === contractAddress.toLowerCase()) {
                            this.logger.debug(`Log ${i} matches contract address, attempting to decode...`);
                            const decoded = this.blockchain.decodeTracingLogs(log);
                            this.logger.debug(`Decoded event - AID: ${decoded.aid}, Action: ${decoded.action}`);

                            if (decoded.action === 'MIGRATED' && decoded.aid === expectedAid) {
                                this.logger.log(`✓ Verified MIGRATED event in transaction receipt for AID: ${expectedAid}`);
                                foundMigratedEvent = true;

                                // Clear timeout and process as success
                                if (this.transactionTimeout) {
                                    clearTimeout(this.transactionTimeout);
                                    this.transactionTimeout = null;
                                }

                                this.handleMigrationSuccess(expectedAid);
                                break;
                            }
                        }
                    } catch (e) {
                        this.logger.debug(`Error decoding log ${i}: ${e.message}`);
                    }
                }
            } else {
                this.logger.warn(`Receipt contains NO logs at all!`);
            }

            if (!foundMigratedEvent) {
                this.logger.warn(`Transaction succeeded but no MIGRATED event found in receipt for AID: ${expectedAid}`);
                this.logger.warn('Will wait for event via WebSocket subscription as fallback');
            }

        } catch (error) {
            this.logger.error(`Error verifying transaction: ${error.message}`);
            this.logger.warn('Will wait for event via WebSocket subscription as fallback');
        }
    }

    private async subscribeToContractEvents(): Promise<void> {
        try {
            this.subscription = await this.blockchain.getTracingEventsSubscription();

            this.subscription.on('data', (log: any) => {
                try {
                    // Log raw event data for debugging
                    this.logger.debug(`Raw blockchain event received - TxHash: ${log.transactionHash}, BlockNumber: ${log.blockNumber}`);

                    const eventObj = this.blockchain.decodeTracingLogs(log);
                    const aid: string = eventObj.aid;
                    const action: string = eventObj.action;

                    this.logger.log(`Blockchain event received - AID: ${aid}, Action: ${action}, TxHash: ${log.transactionHash}`);

                    if (action === 'MIGRATED') {
                        this.handleMigrationSuccess(aid);
                    }
                } catch (error) {
                    this.logger.error(`Error processing blockchain event: ${error.message}`);
                    this.logger.debug(`Raw log data: ${JSON.stringify(log)}`);
                }
            });

            this.subscription.on('error', (error: Error) => {
                this.logger.error(`Blockchain event subscription error: ${error.message}`);
                this.handleMigrationFailure('Event subscription error');
            });

            this.subscription.on('end', () => {
                this.logger.error('Blockchain event subscription ended unexpectedly');
                this.handleMigrationFailure('Event subscription ended');
            });

            this.subscription.on('close', () => {
                this.logger.error('Blockchain event subscription closed unexpectedly');
                this.handleMigrationFailure('Event subscription closed');
            });

            this.logger.log('Successfully subscribed to blockchain events');
        } catch (error) {
            this.logger.error(`Failed to subscribe to blockchain events: ${error.message}`);
            throw error;
        }
    }

    private async processNextRecord(): Promise<void> {
        if (this.currentIndex >= this.totalRecords) {
            this.logger.log('Migration completed successfully!');
            this.cleanup();
            return;
        }

        const record = this.migrationData[this.currentIndex];
        const progress = `${this.currentIndex + 1}/${this.totalRecords}`;

        this.logger.log(`Processing record ${progress} - AID: ${record.aid}`);

        // Check if asset already exists on the blockchain
        try {
            const existingAsset = await this.blockchain.tracingContract.methods.getAsset(record.aid).call();
            if (existingAsset && existingAsset.registered !== '0' && existingAsset.registered !== 0) {
                this.logger.log(`⊘ Asset ${record.aid} already exists on-chain (registered: ${existingAsset.registered}), skipping...`);
                // Move to next record immediately
                this.currentIndex++;
                setTimeout(() => {
                    this.processNextRecord();
                }, 500); // Short delay before next
                return;
            }
        } catch (error) {
            this.logger.warn(`Could not check if asset exists (${error.message}), will attempt migration anyway`);
        }

        // Clear any existing timeout
        if (this.transactionTimeout) {
            clearTimeout(this.transactionTimeout);
        }

        // Set timeout for this transaction
        this.transactionTimeout = setTimeout(() => {
            this.logger.error(`Transaction timeout for AID: ${record.aid} - No MIGRATED event received within ${this.TRANSACTION_TIMEOUT_MS / 1000} seconds`);
            this.logger.warn('This could indicate:');
            this.logger.warn('  1. Transaction failed on-chain but no error was emitted');
            this.logger.warn('  2. WebSocket event subscription was disconnected');
            this.logger.warn('  3. Transaction is still pending in mempool');
            this.logger.warn('  4. Event was emitted but not received due to network issues');
            this.handleMigrationFailure(`Transaction timeout for AID: ${record.aid}`);
        }, this.TRANSACTION_TIMEOUT_MS);

        try {
            // Prepare the contract method call
            const method = this.blockchain.tracingContract.methods.createEntry(
                record.aid,
                record.hash,
                record.tid,
                record.registered,
                record.predecessor,
                record.successor
            ).encodeABI();

            this.logger.debug(`Submitting transaction for AID: ${record.aid}`);
            this.logger.debug(`  Hash: ${record.hash}`);
            this.logger.debug(`  TID: ${record.tid}`);
            this.logger.debug(`  Registered: ${record.registered}`);

            // Execute the transaction
            const successCallback = () => {
                // Extract transaction hash that was attached by BlockchainService
                const txHash = (successCallback as any).__txHash;
                this.currentTransactionHash = txHash;

                this.logger.log(`Transaction submitted successfully for AID: ${record.aid}`);

                // Instead of waiting for WebSocket event (which is unreliable),
                // check the transaction receipt directly
                if (txHash) {
                    this.logger.log(`Checking transaction receipt for hash: ${txHash}`);
                    this.verifyTransactionSuccess(txHash, record.aid);
                } else {
                    this.logger.log(`Waiting for MIGRATED event from blockchain...`);
                }
            };

            this.blockchain.executeTransaction(
                this.blockchain.tracingContract,
                method,
                successCallback,
                () => {
                    this.logger.error(`Transaction failed for AID: ${record.aid}`);
                    if (this.transactionTimeout) {
                        clearTimeout(this.transactionTimeout);
                        this.transactionTimeout = null;
                    }
                    this.handleMigrationFailure(`Transaction failed for AID: ${record.aid}`);
                }
            );
        } catch (error) {
            this.logger.error(`Error processing record ${progress}: ${error.message}`);
            if (this.transactionTimeout) {
                clearTimeout(this.transactionTimeout);
                this.transactionTimeout = null;
            }
            this.handleMigrationFailure(`Error processing record: ${error.message}`);
        }
    }

    private handleMigrationSuccess(aid: string): void {
        const currentRecord = this.migrationData[this.currentIndex];

        if (currentRecord && currentRecord.aid === aid) {
            // Clear the transaction timeout since we received the event
            if (this.transactionTimeout) {
                clearTimeout(this.transactionTimeout);
                this.transactionTimeout = null;
            }

            const progress = `${this.currentIndex + 1}/${this.totalRecords}`;
            this.logger.log(`Migration successful for record ${progress} - AID: ${aid}`);

            // Move to next record
            this.currentIndex++;

            // Process next record after a short delay to avoid overwhelming the system
            setTimeout(() => {
                this.processNextRecord();
            }, 2000);
        } else {
            this.logger.warn(`Received MIGRATED event for unexpected AID: ${aid}`);
        }
    }

    private handleMigrationFailure(errorMessage: string): void {
        this.logger.error(`Migration failed: ${errorMessage}`);
        this.cleanup();
        process.exit(1);
    }

    private cleanup(): void {
        if (this.subscription) {
            this.subscription.unsubscribe();
            this.subscription = null;
            this.logger.log('Unsubscribed from blockchain events');
        }
    }

    public getProgress(): { current: number; total: number; percentage: number } {
        return {
            current: this.currentIndex,
            total: this.totalRecords,
            percentage: this.totalRecords > 0 ? Math.round((this.currentIndex / this.totalRecords) * 100) : 0
        };
    }
}

// CLI interface
async function main() {
    const args = process.argv.slice(2);

    if (args.length !== 1) {
        console.error('Usage: npm run migrate <migration-file.json>');
        console.error('Example: npm run migrate ./data/migration-data.json');
        process.exit(1);
    }

    const migrationFile = args[0];
    const migrationUtility = new MigrationUtility();

    try {
        // Load migration data
        await migrationUtility.loadMigrationData(migrationFile);

        // Start migration process
        await migrationUtility.startMigration();

        // Keep the process running to listen for events
        process.on('SIGINT', () => {
            console.log('\nMigration interrupted by user');
            process.exit(0);
        });

    } catch (error) {
        console.error(`Migration failed: ${error.message}`);
        process.exit(1);
    }
}

// Export for potential use as a module
export { MigrationUtility, MigrationRecord, SourceRecord };

// Run CLI if this file is executed directly
if (require.main === module) {
    main();
}