import { Injectable, Logger } from "@nestjs/common";
import { ConfigService } from "@nestjs/config";
import { InjectConnection } from "@nestjs/mongoose";
import { Connection } from "mongoose";
import { OfferingDescriptor } from "../dtos/offering-descriptor";
import { OfferingRecord } from "../dtos/offering-record";
import { OfferingReference } from "src/dtos/offering-reference";
import { BusinessModel } from 'src/constants';

@Injectable()
export class OfferingCatalogue {
    private collection: any;

    constructor(
        @InjectConnection() private connection: Connection,
        private readonly configService: ConfigService
    ) {
        this.collection = this.connection.db.collection(
            this.configService.get<string>("DATABASE_COLLECTION")
        );
    }

    public async insertDraft(oid: string, offering: OfferingDescriptor, patCorrelationId?: string): Promise<void> {
        const document: any = {
            oid: oid,
            offering: offering,
            status: "DRAFT",
        };
        
        if (patCorrelationId) {
            document.patCorrelationId = patCorrelationId;
        }
        
        const record = await this.collection.insertOne(document);
        Logger.debug(`Draft offering with OID ${oid} added to catalogue with id ${record.insertedId}${patCorrelationId ? ` and PAT correlation ID ${patCorrelationId}` : ''}`);
    }

    public async retrieve(oid: string, status?: string): Promise<OfferingRecord> {
        const query: any = { oid };
        if (status) {
            query.status = status;
        }

        const record = await this.collection.findOne(query);
        if (record) {
            const offeringRecord = new OfferingRecord();
            offeringRecord.oid = record.oid;
            offeringRecord.offering = Object.assign(new OfferingDescriptor(), record.offering);
            offeringRecord.status = record.status;
            return offeringRecord;
        } else {
            return null;
        }
    }

    public async listActiveByAsset(aid: string): Promise<OfferingReference[]> {
        const docs = await this.collection
            .find({
                "offering.asset": aid,
                status: "ACTIVE",
            })
            .toArray();

        return this.mapToOfferingReferences(docs);
    }

    public async listActiveByAssets(aids: string[]): Promise<OfferingReference[]> {
        const docs = await this.collection
            .find({
                "offering.asset": { $in: aids },
                status: "ACTIVE",
            })
            .toArray();

        return this.mapToOfferingReferences(docs);
    }

    private mapToOfferingReferences(docs: any[]): OfferingReference[] {
        return docs.map((doc) => {
            const ref = new OfferingReference();
            ref.aid = doc.offering.asset;
            ref.oid = doc.oid;
            if (doc.offering.unit.duration) {
                ref.model = BusinessModel.SUB;
            } else if (doc.offering.unit.retrievals) {
                ref.model = BusinessModel.PAYU;
            } else if (doc.offering.unit.volume) {
                ref.model = BusinessModel.PAYG;
            } else {
                ref.model = BusinessModel.FREE;
            }
            ref.title = doc.offering.title;
            ref.price = doc.offering.price;
            return ref;
        });
    }

    public async setActive(oid: string): Promise<boolean> {
        const result = await this.collection.updateOne(
            { oid: oid, status: 'DRAFT' },
            { $set: { status: 'ACTIVE' } }
        );
        const outcome: boolean = result.modifiedCount > 0;
        outcome ? Logger.debug(`Draft offering with OID ${oid} is now active`) :
            Logger.warn(`Draft offering with OID ${oid} not found in catalogue`);
        return outcome;
    }

    public async setArchived(oid: string): Promise<boolean> {
        const result = await this.collection.updateOne(
            { oid: oid, status: 'ACTIVE' },
            { $set: { status: 'ARCHIVED' } }
        );
        const outcome: boolean = result.modifiedCount > 0;
        outcome ? Logger.debug(`Active offering with OID ${oid} has been archived`) :
            Logger.warn(`Active offering with OID ${oid} not found in catalogue`);
        return outcome;
    }

    public async setArchivedByAsset(aid: string): Promise<number> {
        const result = await this.collection.updateMany(
            {
                "offering.asset": aid,
                status: "ACTIVE"
            },
            { $set: { status: "ARCHIVED" } }
        );
        const count = result.modifiedCount;
        if (count > 0) {
            Logger.debug(`Archived ${count} active offerings for asset ${aid}`);
        } else {
            Logger.debug(`No active offerings found for asset ${aid}`);
        }
        return count;
    }

    public async countActive(): Promise<number> {
        return await this.collection.countDocuments({ status: 'ACTIVE' });
    }


    public async deleteDraft(oid: string): Promise<boolean> {
        const result = await this.collection.deleteOne(
            { oid: oid, status: 'DRAFT' }
        );
        const outcome: boolean = result.deletedCount > 0;
        outcome ? Logger.debug(`Draft offering with OID ${oid} was deleted`) :
            Logger.warn(`Draft offering with OID ${oid} not found in catalogue`);
        return outcome;
    }

    public async getPatCorrelationId(oid: string): Promise<string | null> {
        const record = await this.collection.findOne({ oid }, { projection: { patCorrelationId: 1 } });
        return record?.patCorrelationId || null;
    }

    public async listArchivedByAsset(aid: string): Promise<OfferingRecord[]> {
        const docs = await this.collection
            .find({
                "offering.asset": aid,
                status: "ARCHIVED",
            })
            .toArray();

        return docs.map((doc) => {
            const record = new OfferingRecord();
            record.oid = doc.oid;
            record.offering = Object.assign(new OfferingDescriptor(), doc.offering);
            record.status = doc.status;
            return record;
        });
    }

    public async listActiveOfferingsByAsset(aid: string): Promise<OfferingRecord[]> {
        const docs = await this.collection
            .find({
                "offering.asset": aid,
                status: "ACTIVE",
            })
            .toArray();

        return docs.map((doc) => {
            const record = new OfferingRecord();
            record.oid = doc.oid;
            record.offering = Object.assign(new OfferingDescriptor(), doc.offering);
            record.status = doc.status;
            return record;
        });
    }
}
