import { ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags, ApiBody } from "@nestjs/swagger";
import { BadRequestException, Body, Controller, Get, HttpCode, Param, Put, Query } from "@nestjs/common";
import { OfferingService } from "./offering-service";
import { translateToHttpException } from 'src/utils/generic-utils';
import { OfferingDescriptor } from 'src/dtos/offering-descriptor';
import { OfferingRecord } from 'src/dtos/offering-record';
import { HTTP_ERR_400, HTTP_ERR_404, HTTP_ERR_500 } from 'src/constants';
import { OfferingReference } from 'src/dtos/offering-reference';
import { Identifiers } from 'src/dtos/identifiers';

@ApiTags('Internal API: Offering Catalogue')
@Controller('/pt/v1.0/offerings')
export class OfferingControllerInternal {

    constructor(private readonly offeringsService: OfferingService) { }

    @Get('/:oid')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieve offering',
        description: `Returns the catalogue entry of an active or archived offering.`
    })
    @ApiParam({
        name: 'oid',
        required: true,
        type: String,
        description: 'OID of target offering'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The catalogue entry of the requested offering',
        type: OfferingRecord,
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveOffering(@Param('oid') oid: string): Promise<OfferingRecord> {
        try {
            return await this.offeringsService.retrieveOffering(null, oid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/:oid/confirm')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirm offering',
        description: `Changes the status of an existing entry in the Offering Catalogue from “draft” to “published”.`
    })
    @ApiParam({
        name: 'oid',
        required: true,
        type: String,
        description: 'OID of target offering'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The request for confirmation has been received and will be processed offline'
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async confirmOffering(@Param('oid') oid: string): Promise<void> {
        try {
            await this.offeringsService.confirmOffering(oid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get()
    @HttpCode(200)
    @ApiOperation({
        summary: 'List offerings for asset',
        description: `Returns a list of references pointing to all the active offerings for a given trusted asset.`
    })
    @ApiQuery({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of root asset'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to the offerings linked to the root asset',
        type: OfferingReference,
        isArray: true
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveOfferingsForAsset(@Query('aid') aid: string): Promise<OfferingReference[]> {
        if (typeof aid !== 'string' || aid.trim().length === 0) {
            throw new BadRequestException("Required parameter aid is missing");
        }
        try {
            return await this.offeringsService.listActiveOfferingsForAsset(null, aid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put()
    @HttpCode(200)
    @ApiOperation({
        summary: 'List offerings for multiple assets',
        description: `Returns a list of references pointing to all the active offerings for the provided trusted assets.`
    })
    @ApiBody({
        type: Identifiers,
        description: 'Array of asset identifiers (AIDs)',
        required: true
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to the offerings linked to the requested assets',
        type: OfferingReference,
        isArray: true
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveOfferings(@Body() identifiers: Identifiers): Promise<OfferingReference[]> {
        if (!identifiers || !Array.isArray(identifiers.ids) || identifiers.ids.length === 0) {
            throw new BadRequestException("Required parameter ids must be a non-empty array");
        }
        try {
            return await this.offeringsService.listActiveOfferings(null, identifiers.ids);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/:oid/reject')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Reject offering',
        description: `Deletes an existing entry with “draft” status from the Offering Catalogue.`
    })
    @ApiParam({
        name: 'oid',
        required: true,
        type: String,
        description: 'OID of target offering'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The request for rejection has been received and will be processed offline'
    })
    async rejectOffering(@Param('oid') oid: string): Promise<void> {
        try {
            await this.offeringsService.rejectOffering(oid);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}
