import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags } from "@nestjs/swagger";
import { BadRequestException, Body, Controller, Delete, Get, HttpCode, Logger, Param, Post, Put, Query, UseGuards } from "@nestjs/common";
import { OfferingService } from "./offering-service";
import { OfferingDescriptor } from "../dtos/offering-descriptor";
import { OfferingRecord } from "../dtos/offering-record";
import { AuthGuard } from "../auth/auth-guard";
import { RolesGuard } from "../auth/roles-guard";
import { Identifier } from "src/dtos/identifier";
import { OfferingReference } from "src/dtos/offering-reference";
import { UserInfo } from "src/auth/user-info";
import { User } from "src/auth/user-decorator";
import { Roles } from "src/auth/roles-decorator";
import { ROLE_ADMIN, ROLE_OPERATOR, ROLE_SUPERUSER, ROLE_USER } from 'src/auth/auth-constants';
import { translateToHttpException } from 'src/utils/generic-utils';
import { HTTP_ERR_400, HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_500, HTTP_ERR_502, HTTP_ERR_404 } from 'src/constants';

@ApiTags('Open API: Offering Catalogue')
@Controller('/api/v1.0/offerings')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class OfferingController {

    constructor(private readonly offeringsService: OfferingService) { }

    @Post()
    @HttpCode(202)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Publish offering',
        description: `Launches the process for publishing a new offering on the marketplace for an existing trusted asset,
 by creating a temporary entry in the Offering Catalogue and by communicating the details of the offering to the T&M module
 (first phase). The caller must be affiliated to the same organization that originally published the asset. The process
 will then continue in the background and will be completed asynchronously (second phase) without any further interaction
 by the caller.`
    })
    @ApiBody({
        type: OfferingDescriptor,
        description: 'Offering properties.',
        required: true,
    })
    @ApiQuery({
        name: 'pat',
        required: false,
        description: 'PAT correlation ID for tracking offering lifecycle',
        type: String,
        example: 'abc-123'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The identifier assigned to the offering by the system.',
        type: Identifier,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async publishOffering(@User() user: UserInfo, @Body() offering: OfferingDescriptor, @Query('pat') patCorrelationId?: string): Promise<Identifier> {
        Logger.log('Publishing requested for offering: ' + JSON.stringify(offering));
        Logger.log('Calling user: ' + JSON.stringify(user));
        if (patCorrelationId) {
            Logger.log('PAT correlation ID: ' + patCorrelationId);
        }
        try {
            return await this.offeringsService.publishOffering(user, offering, patCorrelationId);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get()
    @HttpCode(200)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'List offerings for asset',
        description: `Returns a list of references pointing to all the active offerings for a given trusted asset.`
    })
    @ApiQuery({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of root asset'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to the offerings linked to the root asset',
        type: OfferingReference,
        isArray: true
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveOfferingsForAsset(@User() user: UserInfo, @Query('aid') aid: string): Promise<OfferingReference[]> {
        if (typeof aid !== 'string' || aid.trim().length === 0) {
            throw new BadRequestException("Required parameter aid is missing");
        }
        try {
            return await this.offeringsService.listActiveOfferingsForAsset(user, aid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get('/:oid')
    @HttpCode(200)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieve offering',
        description: `Returns the catalogue entry of an active or archived offering.`
    })
    @ApiParam({
        name: 'oid',
        required: true,
        type: String,
        description: 'OID of target offering'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The catalogue entry of the requested offering',
        type: OfferingRecord,
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveOffering(@User() user: UserInfo, @Param('oid') oid: string): Promise<OfferingRecord> {
        try {
            return await this.offeringsService.retrieveOffering(user, oid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/:oid')
    @HttpCode(204)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Unpublish offering',
        description: `Removes a given offering from the marketplace. The caller must either have the administrator role or
 must be affiliated to the same organization that originally published the asset. Note that although this operation does not
 physically delete any record, it cannot be reverted.`
    })
    @ApiParam({
        name: 'oid',
        required: true,
        type: String,
        description: 'The OID of the target offering'
    })
    @ApiResponse({
        status: 204 // No Content
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async unpublishOffering(@User() user: UserInfo, @Param('oid') oid: string): Promise<void> {
        Logger.log('Unpublishing requested for offering with OID ' + oid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            await this.offeringsService.unpublishOffering(user, oid);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}
