import { Injectable, Logger, NotFoundException } from '@nestjs/common';
import { OfferingDescriptor } from '../dtos/offering-descriptor';
import { OfferingRecord } from '../dtos/offering-record';
import { SHA3 } from 'sha3';
import { ConfigService } from '@nestjs/config';
import { GovService } from '../integration/gov-service';
import { TmService } from '../integration/tm-service';
import { OfferingCatalogue } from './offering-catalogue';
import { TradingInfo } from '../dtos/trading-info';
import { Identifier } from 'src/dtos/identifier';
import { OfferingReference } from 'src/dtos/offering-reference';
import { UserInfo } from 'src/auth/user-info';
import { ErrorCondition, isApmDisabled, isAuthDisabled, isFdacDisabled, isPrivilegedUser, isRqueueDisabled, isTmDisabled, isXcheckDisabled, throwExceptionWithErrorCondition, checkPublishingRules } from 'src/utils/generic-utils';
import { MarkdownUtils } from 'src/utils/markdown-utils';
import { FAME_PFX, PERIOD_DENOMS, PERIOD_MILLISECS } from 'src/constants';
import { FdacService } from 'src/integration/fdac-service';
import { PatService } from 'src/integration/pat-service';
import { ApmService } from 'src/integration/apm-service';
const base58check = require('base58check');

@Injectable()
export class OfferingService {

    private readonly skipAuth: boolean = false;
    private readonly skipXCheck: boolean = false;
    private readonly skipFdac: boolean = false;
    private readonly skipTM: boolean = false;
    private readonly skipApm: boolean = false;
    private readonly skipRQueue: boolean = false;

    constructor(
        private readonly config: ConfigService,
        private readonly ocat: OfferingCatalogue,
        private readonly fdac: FdacService,
        private readonly tm: TmService,
        private readonly apm: ApmService,
        private readonly gov: GovService,
        private readonly pat: PatService,
    ) {
        this.skipAuth = isAuthDisabled(this.config);
        this.skipXCheck = isXcheckDisabled(this.config);
        this.skipFdac = isFdacDisabled(this.config);
        this.skipTM = isTmDisabled(this.config);
        this.skipApm = isApmDisabled(this.config);
        this.skipRQueue = isRqueueDisabled(this.config);
    }

    public async publishOffering(user: UserInfo, offering: OfferingDescriptor, patCorrelationId?: string, isCloning: boolean = false): Promise<Identifier> {
        const aid: string = offering.asset;

        // Validation checks
        if (this.skipXCheck || this.skipFdac) {
            Logger.debug('FDAC and/or offering x-check are disabled, skipping all offering rules verification');
        } else {
            // check that catalogue entry exists, that the user is affiliated to the asset publisher and
            // that the beneficiary account is valid for the asset publisher
            try {
                await checkPublishingRules(this.fdac, this.apm, this.gov, aid, user, this.skipAuth, offering.tid);
            } catch (err) {
                if (isCloning) {
                    Logger.error(`Validation failed for cloned offering: ${err.message}`);
                    return new Identifier(null); // Return Identifier with null id when cloning fails
                } else {
                    throw err; // Re-throw for normal publishing
                }
            }
        }

        // Process Markdown fields: convert safe Markdown to HTML
        // Skip conversion if the offering is being cloned (already contains HTML)
        if (!isCloning) {
            if (offering.license) {
                offering.license = MarkdownUtils.markdownToHtml(offering.license);
                Logger.debug(`Converting Markdown to HTML for offering license field: ` + offering.license);
            }
            if (offering.sla) {
                offering.sla = MarkdownUtils.markdownToHtml(offering.sla);
                Logger.debug(`Converting Markdown to HTML for offering SLA field: ` + offering.sla);
            }
        }

        // generate a unique identifier for the offering, which is also an integrity seal
        // of the entire offering descriptor (except the OID field)
        const hash = new SHA3(256);
        hash.update(JSON.stringify(offering));
        let oid: string = base58check.encode(hash.digest('hex'));

        let tradingInfo: TradingInfo = null;
        try {
            tradingInfo = this.createTradeSetupInfo(oid, offering);
        } catch (err) {
            Logger.error(`Error extracting trading info from input for OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            if (isCloning) {
                return new Identifier(null); // Return Identifier with null id when cloning fails
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not extract trading info from input');
            }
        }
        Logger.debug('Trading info: ' + JSON.stringify(tradingInfo));

        // we insert the offering in the catalogue as a draft that will be later confirmed or rejected by T&M
        // this is a required step, so it is synchronous and exceptions are fatal (or null for cloning)
        try {
            await this.ocat.insertDraft(oid, offering, isCloning ? undefined : patCorrelationId); // debug logging is done by the catalogue service
        } catch (err) {
            Logger.error(`Error creating the catalogue entry for offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            if (isCloning) {
                return new Identifier(null); // Return Identifier with null id when cloning fails
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not write to the offering catalogue');
            }
        }

        // User request queue management - skip when cloning
        if (!isCloning) {
            if (this.skipAuth || this.skipRQueue) {
                Logger.debug('Authentication and/or RQUEUE are disabled: no user request management');
            } else {
                // we do this _before_ calling TM because if TM fails there's no user request to update
                try {
                    // add a user request to the queue
                    // this is a NOT a required step, so it is asynchronous and exceptions swallowed
                    this.gov.enqueueRequest(oid, user.uid, '', 'Processing offering publishing request');
                } catch (err) {
                    // should never happen as exceptions are managed by the queue service
                    Logger.error(`Error adding user request to the queue: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                }
            }
        }

        // Call T&M service to set up trading
        if (this.skipTM) {
            Logger.debug('T&M is disabled');
        } else {
            try {
                await this.tm.setupTrading(tradingInfo);
            } catch (err) {
                Logger.error(`Error calling the T&M module for the offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                if (isCloning) {
                    // Best effort to clean up
                    this.ocat.deleteDraft(oid).catch((e) => {
                        Logger.warn(`Failed to delete draft offering ${oid} after TM error: ${e.message}`);
                    });
                    return new Identifier(null); // Return Identifier with null id when cloning fails
                } else {
                    this.rejectOffering(oid); // best effort to remove the draft entry from the catalogue, asynchronously
                    throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error calling the T&M module');
                }
            }
        }

        Logger.log(`First phase of ${isCloning ? 'cloning' : 'publishing'} for offering with OID ${oid} completed`);

        return new Identifier(oid);
    }

    public async unpublishOffering(user: UserInfo, oid: string): Promise<void> {
        const offeringRecord: OfferingRecord = await this.ocat.retrieve(oid, 'ACTIVE');
        if (!offeringRecord) {
            Logger.warn(`Active offering with OID ${oid} not found in the catalogue`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Offering does not exist or is not published');
        }

        if (this.skipXCheck || this.skipFdac) {
            Logger.debug('FDAC and/or offering x-check are disabled, skipping all offering rules verification');
        } else {
            // check that the user is affiliated to the asset publisher
            await checkPublishingRules(this.fdac, this.apm, this.gov, offeringRecord.offering.asset, user, this.skipAuth);
        }

        try {
            await this.ocat.setArchived(oid); // debug logging is done by the catalogue service
        } catch (err) {
            Logger.error(`Error updating the catalogue entry for offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not write to the offering catalogue');
        }

        if (this.skipTM) {
            Logger.debug('T&M is disabled');
        } else {
            try {
                await this.tm.disableTrading(oid);
            } catch (err) {
                Logger.error(`Error calling the T&M module for the offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error calling the T&M module');
            }
        }

        Logger.log('Offering with OID ' + oid + ' unpublished successfully');
    }


    public async retrieveOffering(user: UserInfo, oid: string): Promise<OfferingRecord> {
        // Only return ACTIVE and ARCHIVED offerings, exclude DRAFT
        const offeringRecord = await this.ocat.retrieve(oid);
        if (!offeringRecord || offeringRecord.status === 'DRAFT') {
            Logger.warn(`Offering with OID ${oid} not found in the catalogue or is in DRAFT status`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Offering does not exist or is not accessible');
        }

        if (this.skipApm || user == null) {
            Logger.debug('No policies are applied to offering with OID ' + oid);
        } else if (!isPrivilegedUser(user)) {
            const aid = offeringRecord.offering.asset;
            if (!await this.apm.checkVisibility(aid, user)) {
                Logger.debug('Access to offering with OID ' + oid + ' denied to user with UID ' + user.uid);
                throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
            }
        }

        return offeringRecord;
    }

    public async listActiveOfferingsForAsset(user: UserInfo, aid: string): Promise<OfferingReference[]> {
        if (this.skipApm || user == null) {
            Logger.debug('No policies are applied to asset with AID ' + aid);
        } else if (!isPrivilegedUser(user)) {
            if (!await this.apm.checkVisibility(aid, user)) {
                Logger.debug('Access to asset with AID ' + aid + ' denied to user with UID ' + user.uid);
                throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Catalogue entry could not be found');
            }
        }

        return this.ocat.listActiveByAsset(aid);
    }

    public async listActiveOfferings(user: UserInfo, aids: string[]): Promise<OfferingReference[]> {
        // Filter assets based on visibility permissions
        let visibleAids: string[] = aids;

        if (!this.skipApm && user != null && !isPrivilegedUser(user)) {
            // Check visibility for each asset and filter out inaccessible ones
            const visibilityChecks = await Promise.all(
                aids.map(async (aid) => {
                    try {
                        const isVisible = await this.apm.checkVisibility(aid, user);
                        if (!isVisible) {
                            Logger.debug('Access to asset with AID ' + aid + ' denied to user with UID ' + user.uid);
                        }
                        return { aid, isVisible };
                    } catch (err) {
                        Logger.warn(`Error checking visibility for asset ${aid}: ${err.message}`);
                        return { aid, isVisible: false };
                    }
                })
            );

            visibleAids = visibilityChecks
                .filter(check => check.isVisible)
                .map(check => check.aid);

            Logger.debug(`Filtered ${aids.length} assets to ${visibleAids.length} visible assets for user ${user.uid}`);
        }

        // If no visible assets, return empty array
        if (visibleAids.length === 0) {
            return [];
        }

        // Retrieve all offerings for visible assets in a single query
        return this.ocat.listActiveByAssets(visibleAids);
    }

    public async confirmOffering(oid: string): Promise<void> {
        Logger.log('Confirmation requested for offering with OID ' + oid);
        try {
            this.ocat.setActive(oid)
                .then((result: boolean) => {
                    if (result) {
                        Logger.log('Second phase of publishing for offering with OID ' + oid + ' completed');
                        // Call PAT service if correlation ID exists
                        this.notifyPatService(oid);
                    } else {
                        Logger.error('Second phase of publishing for offering with OID ' + oid + ' FAILED');
                    }
                    this.finalizeRequest(oid, result);
                })
                .catch((error) => {
                    Logger.error('Second phase of publishing for offering with OID ' + oid + ' FAILED: ', error);
                    this.finalizeRequest(oid, false);
                });
        } catch (err) {
            Logger.error(`Error activating the catalogue entry for offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not write to the offering catalogue');
        }
    }

    public async rejectOffering(oid: string): Promise<void> {
        Logger.log('Rejection requested for offering with OID ' + oid);
        // best effort at performing the required operations, but exceptions are swallowed
        this.ocat.deleteDraft(oid).catch((err) => {
            Logger.warn(`Error deleting draft offering with OID ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
        });
        this.finalizeRequest(oid, false);
    }

    private createTradeSetupInfo(oid: string, offering: OfferingDescriptor): TradingInfo {
        const tradingSetupInfo = new TradingInfo();
        tradingSetupInfo.assetid = offering.asset
        tradingSetupInfo.oid = oid;
        tradingSetupInfo.beneficiary = offering.tid;
        tradingSetupInfo.resource = offering.tid;
        tradingSetupInfo.price = offering.price;
        tradingSetupInfo.cds_target = JSON.parse(offering.cdsinst);

        if (offering.unit) {
            if (offering.unit.retrievals)
                tradingSetupInfo.cap_downloads = offering.unit.retrievals + '';
            else if (offering.unit.volume)
                tradingSetupInfo.cap_volume = offering.unit.volume;
            else if (offering.unit.duration) {
                // duration is expressed as <number><period>, where period is one of m, H, D, M, Y
                const duration: string = offering.unit.duration.trim()
                for (let i = 0; i < PERIOD_DENOMS.length; i++) {
                    if (duration.endsWith(PERIOD_DENOMS[i])) {
                        const units: number = +duration.substring(0, duration.indexOf(PERIOD_DENOMS[i]));
                        const unitSize: number = PERIOD_MILLISECS[i];
                        tradingSetupInfo.cap_duration = (units * unitSize).toString(); // save as string if needed
                        break;
                    }
                }
            }
        }

        return tradingSetupInfo;
    }

    private async notifyPatService(oid: string): Promise<void> {
        try {
            const patCorrelationId = await this.ocat.getPatCorrelationId(oid);
            if (patCorrelationId) {
                Logger.debug(`Notifying PAT service for offering ${oid} with correlation ID ${patCorrelationId}`);
                await this.pat.sendConfirmation(patCorrelationId, oid);
                Logger.debug(`PAT notification sent successfully for offering ${oid}`);
            }
        } catch (error) {
            // Fail-safe: log error but don't throw
            Logger.warn(`Failed to notify PAT service for offering ${oid}: ${error.message}`);
        }
    }

    private finalizeRequest(oid: string, success: boolean) {
        if (this.skipAuth || this.skipRQueue) {
            Logger.debug('Authentication and/or RQUEUE are disabled: no user request management');
        } else {
            try {
                success ? this.gov.confirmRequest(oid, 'Offering was published with OID ' + oid) :
                    this.gov.rejectRequest(oid, 'Offering publishing failed');
            } catch (err) {
                // should never happen as exceptions are managed by the queue service
                Logger.error(`Error updating user request with id ${oid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
        }
    }
}
