import { Body, Controller, Delete, Get, HttpCode, Logger, NotFoundException, Param, Post } from "@nestjs/common";
import { ApiBody, ApiOperation, ApiParam, ApiResponse, ApiTags } from "@nestjs/swagger";
import { PermissionService } from "./permission-service";
import { AccountIdentifier } from 'src/dtos/account-identifier';
import { HTTP_ERR_400, HTTP_ERR_500, HTTP_ERR_404 } from 'src/constants';
import { translateToHttpException } from 'src/utils/generic-utils';

@ApiTags('Internal API: Blockchain Permissions')
@Controller('/pt/v1.0/permissions')
export class PermissionControllerInternal {

    constructor(private readonly permissionService: PermissionService) { }

    @Post()
    @HttpCode(202)
    @ApiOperation({
        summary: 'Grant permission',
        description: `Launches the process for granting blockchain permission to a trading account, by submitting a
 blockchain transaction that will add the given account to the on-chain list of "permissioned" accounts, if not already
 present. Once successfully launched, the process will continue in the background and will be completed without any further
 interaction by the caller. Note that if the account is already in the list, the transaction will still succeed but will
 have no effects. This method is only used internally, as part of the P&T/GOV integration: when the blockchain transaction
 is confirmed, a specific GOV callback service endpoint will be called.`,
    })
    @ApiBody({
        type: AccountIdentifier,
        description: `The "tid" attribute must contain the TID of the trading account to be granted permission.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The permission granting process has been successfully launched. Check later for the status of the account.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async grantPermission(@Body() id: AccountIdentifier): Promise<void> {
        Logger.log('Permission requested (internal call) for account with TID ' + id.tid);
        return this.permissionService.setPermission(id.tid, true);
    }

    @Get('/:tid')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Checks permission',
        description: `Checks is a given trading account has received blockchain permission. If that is the case, the
  operation will end successfully with a 204 NO CONTENT status code; otherwise, it will fail with a 404 NOT FOUND
  status code. In both cases, the response body is empty.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The target'
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async checkPermission(@Param('tid') tid: string): Promise<void> {
        try {
            const hasPermission = await this.permissionService.hasPermission(tid);
            if (!hasPermission) {
                throw new NotFoundException('Trading account with TID ' + tid + ' not found in allowlist');
            }
        } catch (err) {
            // Only translate errors that come from the service (with the special format)
            if (err instanceof NotFoundException) {
                throw err; // Re-throw manual exceptions as-is
            }
            translateToHttpException(err);
        }
    }

    @Delete('/:tid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Revoke permission',
        description: `Launches the process for revoking blockchain permission to a trading account, by submitting a
 blockchain transaction that will remove the given account from the on-chain list of "permissioned" accounts, if present.
 Once successfully launched, the process will continue in the background and will be completed without any further
 interaction by the caller. Note that if the account is not in the list, the transaction will still succeed but will
 have no effects. This method is only used internally, as part of the P&T/GOV integration: when the blockchain transaction
 is confirmed, a specific GOV callback service endpoint will be called.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account',
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The permission revocation process has been successfully launched. Check later for the status of the account.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async revokePermission(@Param('tid') tid: string): Promise<void> {
        Logger.log('Permission revocation requested (internal call) for account with TID ' + tid);
        return this.permissionService.setPermission(tid, false);
    }
}