import { Body, Controller, Delete, Get, HttpCode, Logger, NotFoundException, Param, Post, UseGuards } from "@nestjs/common";
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiResponse, ApiTags } from "@nestjs/swagger";
import { PermissionService } from "./permission-service";
import { AccountIdentifier } from 'src/dtos/account-identifier';
import { HTTP_ERR_400, HTTP_ERR_500, HTTP_ERR_404, HTTP_ERR_401, HTTP_ERR_403 } from 'src/constants';
import { AuthGuard } from 'src/auth/auth-guard';
import { RolesGuard } from 'src/auth/roles-guard';
import { ROLE_ADMIN } from 'src/auth/auth-constants';
import { Roles } from 'src/auth/roles-decorator';
import { User } from 'src/auth/user-decorator';
import { UserInfo } from 'src/auth/user-info';
import { translateToHttpException } from 'src/utils/generic-utils';

@ApiTags('Open API: Blockchain Permissions')
@Controller('/api/v1.0/permissions')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class PermissionController {

    constructor(private readonly permissionService: PermissionService) { }

    @Post()
    @HttpCode(202)
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Grant permission',
        description: `Launches the process for granting blockchain permission to a trading account, by submitting a
 blockchain transaction that will add the given account to the on-chain list of "permissioned" accounts, if not already
 present. Once successfully launched, the process will continue in the background and will be completed without any further
 interaction by the caller. Note that if the account is already in the list, the transaction will still succeed but will
 have no effects. This method can only be used by administrators.`,
    })
    @ApiBody({
        type: AccountIdentifier,
        description: `The "tid" attribute must contain the TID of the trading account to be granted permission.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The permission granting process has been successfully launched. Check later for the status of the account.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async grantPermission(@User() user: UserInfo, @Body() id: AccountIdentifier): Promise<void> {
        Logger.log('Permission requested for account with TID ' + id.tid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        return this.permissionService.setPermission(id.tid, true);
    }

    @Get('/:tid')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Checks permission',
        description: `Checks is a given trading account has received blockchain permission. If that is the case, the
  operation will end successfully with a 204 NO CONTENT status code; otherwise, it will fail with a 404 NOT FOUND
  status code. In both cases, the response body is empty.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The target'
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async checkPermission(@Param('tid') tid: string): Promise<void> {
        try {
            const hasPermission = await this.permissionService.hasPermission(tid);
            if (!hasPermission) {
                throw new NotFoundException('Trading account with TID ' + tid + ' not found in allowlist');
            }
        } catch (err) {
            // Only translate errors that come from the service (with the special format)
            if (err instanceof NotFoundException) {
                throw err; // Re-throw manual exceptions as-is
            }
            translateToHttpException(err);
        }
    }

    @Delete('/:tid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Revoke permission',
        description: `Launches the process for revoking blockchain permission to a trading account, by submitting a
 blockchain transaction that will remove the given account from the on-chain list of "permissioned" accounts, if present.
 Once successfully launched, the process will continue in the background and will be completed without any further
 interaction by the caller. Note that if the account is not in the list, the transaction will still succeed but will
 have no effects. This method can only be used by administrators.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account',
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The permission revocation process has been successfully launched. Check later for the status of the account.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async revokePermission(@User() user: UserInfo, @Param('tid') tid: string): Promise<void> {
        Logger.log('Permission revocation requested for account with TID ' + tid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        return this.permissionService.setPermission(tid, false);
    }
}