import { ConfigService } from "@nestjs/config";
import { BlockchainService } from "src/integration/blockchain-service";
import { Injectable, Logger } from '@nestjs/common';
import axios, { AxiosInstance } from 'axios';
import { ErrorCondition, throwExceptionWithErrorCondition } from 'src/utils/generic-utils';

@Injectable()
export class PermissionService {

    private readonly govProxy: AxiosInstance = null;
    private readonly contract: any = null;

    constructor(
        private readonly config: ConfigService,
        private readonly blockchain: BlockchainService,
    ) {
        this.govProxy = axios.create({
            baseURL: this.config.get<string>('GOV_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
        this.contract = blockchain.permissionContract;
    }

    public setPermission(tid: string, allow: boolean) { // removeAccount
        const method = allow ?
            this.contract.methods.addAccount(tid).encodeABI() :
            this.contract.methods.removeAccount(tid).encodeABI();
        this.blockchain.executeTransaction(
            this.contract,
            method,
            this.handleConfirmation(tid, allow), // callback function on success, that calls the GOV internal API
            () => { // callback function on failure, that logs a warning message
                if (allow) {
                    Logger.warn(`Transaction failed: account with TID ${tid} could not be added to allowlit`);
                } else {
                    Logger.warn(`Transaction failed: account with TID ${tid} could not be removed from allowlit`);
                }
            }
        );
    }

    public async hasPermission(tid: string): Promise<boolean> {
        try {
            return await this.contract.methods.accountPermitted(tid).call();
        } catch (err) {
            Logger.error('Error calling contract method accountPermitted()', err);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error calling the Permission contract')
        }
    }

    // returns a callback function that captures the lexical scope (parameter values and private members)
    private handleConfirmation(tid: string, allow: boolean): () => void {
        return () => {
            let suffix: string = null;
            if (allow) {
                suffix = '/allowed';
                Logger.log(`Transaction confirmed: account with TID ${tid} was added to the blockchain allowlist`);
            } else {
                suffix = '/disallowed';
                Logger.log(`Transaction confirmed: account with TID ${tid} was removed from the blockchain allowlist`);
            }
            try {
                this.govProxy.put('/accounts/' + tid + suffix).then((response) => {
                    if (response.status === 202) {
                        Logger.debug('Call to GOV callback service successful for TID ' + tid);
                    } else {
                        // the call to the GOV callback service failed: log the response
                        Logger.error('Call to the GOV callback service FAILED for TID ' + tid + ', status code is ' + response.status);
                    }
                }).catch((err) => {
                    // we failed to call the GOV callback service: log and swallow the error
                    Logger.error(`Call to the GOV callback service FAILED for TID ${tid} - ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                });
            } catch (err) {
                // we failed to call the GOV callback service: log and swallow the error
                Logger.error(`Error setting up the call to the GOV callback service for TID ${tid} - ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
        };
    }

}   
