import { BadRequestException, Body, Controller, Delete, Get, HttpCode, Logger, Param, Post, Query } from '@nestjs/common';
import { ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags, } from '@nestjs/swagger';
import { SourceService } from './source-service';
import { SourceReference } from 'src/dtos/source-reference';
import { Source } from 'src/dtos/source';
import { Identifier } from 'src/dtos/identifier';
import { LegalEntity } from 'src/dtos/legal-entity';
import { EntityStatus, HTTP_ERR_400, HTTP_ERR_404, HTTP_ERR_500 } from 'src/constants';

@ApiTags('Internal API: Trusted Sources')
@Controller('/pt/v1.0')
export class SourceControllerInternal {

    constructor(private readonly sourceService: SourceService) { }

    @Post('/sources')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Register source',
        description: `Launches the process for the registration a trusted source, as part of the member onboarding workflow
 that is initiated by the GOV module. The registration starts with the submission of a blockchain transaction that will
 create, when the transaction is confirmed, a new Provenance Ledger record with the given unique identifier (PID). The
 process runs in the background: when it completes successfully, the GOV module will be notified.`
    })
    @ApiBody({
        type: Source,
        description: 'Onboarding PID of the trusted source and contact information of the legal entity.',
        required: true,
    })
    @ApiResponse({
        status: 202 // Accepted
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    registerSource(@Body() source: Source) {
        Logger.log('Registration requested for source: ' + JSON.stringify(source));
        this.sourceService.registerSource(source.pid, source.descriptor);
    }

    @Get('/sources')
    @ApiOperation({
        summary: 'List sources',
        description: `Returns a list of references pointing to all the trusted source records, regardless of status, in the Provenance Ledger.
 The caller must have the operator or administrator role.`
    })
    @ApiQuery({
        name: 'status',
        required: false,
        enum: EntityStatus,
        description: `Status to filter members by, accepted values: [${Object.values(EntityStatus).join(', ')}]`,
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to trusted sources',
        type: SourceReference,
        isArray: true
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listSources(@Query('status') status: EntityStatus): Promise<SourceReference[]> {
        if (status) {
            if (status === EntityStatus.A) {
                return await this.sourceService.listSources(true, false);
            } else if (status === EntityStatus.I || status === EntityStatus.D) {
                return await this.sourceService.listSources(false, true);
            } else {
                throw new BadRequestException('Unsupported value in "status"');
            }
        } else {
            return await this.sourceService.listSources(true, true);
        }
    }

    @Get('/sources/:pid')
    @ApiOperation({
        summary: 'Retrieve source',
        description: `Returns a trusted source entry from the Provenance Ledger, regardless of its status.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'The PID of the target trusted source',
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The Provenance Ledger record of the trusted source',
        type: Source,
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveSource(@Param('pid') pid: string): Promise<Source> {
        return await this.sourceService.retrieveSource(pid, false);
    }

    @Delete('/sources/:pid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Deregister source',
        description: `Launches the process for unregistering a trusted source, by submitting a blockchain transaction that will
 set the status of the target record in the Provenance Ledger to “inactive”. The process runs in the background: when it completes
 successfully, the GOV module will be notified. Note that this change of status cannot be reverted.`,
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'The PID of the target trusted source',
    })
    @ApiResponse({
        status: 202 // Accepted
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    deregisterSource(@Param('pid') pid: string) {
        Logger.log('Deregistration requested for source with PID ' + pid);
        this.sourceService.deregisterSource(pid);
    }

    @Get('/active-sources')
    @ApiOperation({
        summary: 'List onboarded sources',
        description: `Returns a list of items representing all the active trusted source entries in the Provenance Ledger.`
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to trusted sources',
        type: SourceReference,
        isArray: true
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveSources(): Promise<SourceReference[]> {
        return await this.sourceService.listSources(true, false);
    }

    @Get('/active-sources/:pid')
    @ApiOperation({
        summary: 'Retrieve onboarded source',
        description: `Returns an active trusted source entry from the Provenance Ledger.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'The PID of the target trusted source',
    })
    @ApiResponse({
        status: 200, // OK
        description: 'A summary of the trusted source information',
        type: SourceReference,
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveActiveSource(@Param('pid') pid: string): Promise<SourceReference> {
        const entry: Source = await this.sourceService.retrieveSource(pid, true); // exception if not found
        const result: SourceReference = new SourceReference();
        result.pid = entry.pid;
        result.name = entry.descriptor.legalName;
        result.in = entry.in;
        result.active = true;
        return result;
    }
}
