import { Controller, Get, Param, } from '@nestjs/common';
import { ApiOperation, ApiParam, ApiResponse, ApiTags, } from '@nestjs/swagger';
import { SourceService } from './source-service';
import { SourceReference } from 'src/dtos/source-reference';
import { Source } from 'src/dtos/source';
import { HTTP_ERR_500, HTTP_ERR_404 } from 'src/constants';

@ApiTags('Open API: Trusted Sources')
@Controller('/api/v1.0')
export class SourceControllerOpen {

    constructor(private readonly sourceService: SourceService) { }

    @Get('/active-sources')
    @ApiOperation({
        summary: 'List onboarded sources',
        description: `Returns a list of items representing all the active trusted source entries in the Provenance Ledger.`
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to trusted sources',
        type: SourceReference,
        isArray: true
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveSources(): Promise<SourceReference[]> {
        return await this.sourceService.listSources(true, false);
    }

    @Get('/active-sources/:pid')
    @ApiOperation({
        summary: 'Retrieve onboarded source',
        description: `Returns an active trusted source entry from the Provenance Ledger.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'The PID of the target trusted source',
    })
    @ApiResponse({
        status: 200, // OK
        description: 'A summary of the trusted source information',
        type: SourceReference,
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveActiveSource(@Param('pid') pid: string): Promise<SourceReference> {
        const entry: Source = await this.sourceService.retrieveSource(pid, true); // exception if not found
        const result: SourceReference = new SourceReference();
        result.pid = entry.pid;
        result.name = entry.descriptor.legalName;
        result.in = entry.in;
        result.active = true;
        return result;
    }
}
