import { BadRequestException, Controller, Get, Param, Query, UseGuards } from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags, } from '@nestjs/swagger';
import { SourceService } from './source-service';
import { AuthGuard } from 'src/auth/auth-guard';
import { SourceReference } from 'src/dtos/source-reference';
import { Source } from 'src/dtos/source';
import { ROLE_OPERATOR, ROLE_ADMIN } from 'src/auth/auth-constants';
import { Roles } from 'src/auth/roles-decorator';
import { RolesGuard } from 'src/auth/roles-guard';
import { HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_500, HTTP_ERR_404 } from 'src/constants';

@ApiTags('Open API: Trusted Sources')
@Controller('/api/v1.0')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class SourceController {

    constructor(private readonly sourceService: SourceService) { }

    @Get('/sources')
    @Roles(ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'List sources',
        description: `Returns a list of references pointing to all the trusted source records, regardless of status, in the Provenance Ledger.
 The caller must have the operator or administrator role.`
    })
    @ApiQuery({
        name: 'status',
        required: false,
        type: String,
        description: `'A' to include only entries with active status, 'I' to include only entries with inactive status,
  omit to include all entries regardless of status`
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to trusted sources',
        type: SourceReference,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listSources(@Query('status') status: string): Promise<SourceReference[]> {
        if (status) {
            if (status.toUpperCase() == 'A') {
                return await this.sourceService.listSources(true, false);
            } else if (status.toUpperCase() === 'I') {
                return await this.sourceService.listSources(false, true);
            } else {
                throw new BadRequestException('Unsupported value in "status"');
            }
        } else {
            return await this.sourceService.listSources(true, true);
        }
    }

    @Get('/sources/:pid')
    @Roles(ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieve source',
        description: `Returns the full details of a given trusted source, including its metadata and status, from the
 Provenance Ledger. The caller must have the operator or administrator role.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'The PID of the target trusted source',
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The Provenance Ledger record of the trusted source',
        type: Source,
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveSource(@Param('pid') pid: string): Promise<Source> {
        return await this.sourceService.retrieveSource(pid, false);
    }
}
