import { Injectable, InternalServerErrorException, Logger, NotFoundException } from '@nestjs/common';
import { BlockchainService } from "../integration/blockchain-service";
import { LegalEntity } from 'src/dtos/legal-entity';
import { Source } from 'src/dtos/source';
import { getISOTimeFromString, safeParseInt } from 'src/utils/generic-utils';
import { SourceReference } from 'src/dtos/source-reference';
import { plainToInstance } from 'class-transformer';
import { validateSync } from 'class-validator';
import { ConfigService } from '@nestjs/config';
import axios, { AxiosInstance } from 'axios';

@Injectable()
export class SourceService {

    private readonly govProxy: AxiosInstance = null;
    private readonly contract: any = null;

    constructor(
        private readonly config: ConfigService,
        private readonly blockchain: BlockchainService
    ) {
        this.govProxy = axios.create({
            baseURL: this.config.get<string>('GOV_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
        this.contract = blockchain.provenanceContract;
    }

    // the blockchain transaction is executed in a fire-and-forget pattern:
    // no need to make the method asynchronous
    public registerSource(pid: string, source: LegalEntity) {
        this.blockchain.executeTransaction(
            this.contract,
            this.contract.methods.registerSource(pid, JSON.stringify(source)).encodeABI(),
            this.handleConfirmation(pid, true), // callback function on success, that calls the GOV internal API
            () => { // callback function on failure, that logs a warning message
                Logger.warn(`Transaction failed: source with PID ${pid} could not be registered`);
            }
        );
    }

    // the blockchain transaction is executed in a fire-and-forget pattern:
    // no need to make the method asynchronous
    public deregisterSource(pid: string) {
        this.blockchain.executeTransaction(
            this.contract,
            this.contract.methods.deactivateSource(pid).encodeABI(),
            this.handleConfirmation(pid, false), // callback function on success, that calls the GOV internal API
            () => { // callback function on failure, that logs a warning message
                Logger.warn(`Transaction failed: source with PID ${pid} could not be unregistered`);
            }
        );
    }

    public async retrieveSource(pid: string, activeOnly: boolean): Promise<Source> {
        let source: Source = null;
        let rec: any = await this.getSourceRecord(pid);
        if (rec) {
            const active = getISOTimeFromString(rec.deactivated).length === 0;
            if (active || !activeOnly) {
                const organization = this.parseDescriptor(rec.descriptor);
                if (organization) {
                    source = new Source();
                    source.pid = rec.pid;
                    source.descriptor = organization;
                    source.in = getISOTimeFromString(rec.registered);
                    source.out = getISOTimeFromString(rec.deactivated);
                } else {
                    Logger.warn(`Source record with PID ${rec.pid} ignored: invalid legal entity descriptor`);
                }
            }
        }

        if (source) {
            return source;
        } else {
            throw new NotFoundException('Item with PID ' + pid + ' was not found');
        }
    }

    public async listSources(onboarded: boolean, offboarded: boolean): Promise<SourceReference[]> {
        const results: SourceReference[] = [];
        if (!onboarded && !offboarded) return results; // short answer to stupid questions

        let records: any;
        try {
            records = await this.contract.methods.getSources().call();
        } catch (err) {
            Logger.error('Error calling contract method getSources()', err);
            throw new InternalServerErrorException('Error calling the Provenance Ledger contract');
        }

        for (const rec of records) {
            const active = getISOTimeFromString(rec.deactivated).length === 0;
            if (active && !onboarded) continue; // they don't want onboarded sources: skip
            if (!active && !offboarded) continue; // they don't want offboarded sources: skip
            const organization = this.parseDescriptor(rec.descriptor);
            if (organization) {
                const item = new SourceReference();
                item.pid = rec.pid;
                item.name = organization.legalName;
                item.in = getISOTimeFromString(rec.registered);
                item.active = active;
                results.push(item);
            } else {
                Logger.warn(`Source record with PID ${rec.pid} skipped: invalid legal entity descriptor`);
                continue;
            }
        }
        return results;
    }

    private async getSourceRecord(pid: string): Promise<any> {
        let rec: any = null;
        try {
            rec = await this.contract.methods.getSource(pid).call();
        } catch (err) {
            Logger.error('Error calling contract method getSource()', err);
            throw new InternalServerErrorException('Error calling the Provenance Ledger contract');
        }
        if (safeParseInt(rec.registered) > 0) { // check if this record exists on the ledger
            return rec;
        } else {
            return null;
        }
    }

    private parseDescriptor(descriptor: string): LegalEntity {
        let obj: LegalEntity = null;
        try {
            obj = plainToInstance(LegalEntity, JSON.parse(descriptor));
            const errors = validateSync(obj); // apply the rules defined in DTO decorators
            if (errors.length > 0) {
                Logger.warn(`Validation failed for legal entity descriptor: ${descriptor}`, errors);
                obj = null;
            }
        } catch (err) {
            Logger.warn(`Parsing failed for legal entity descriptor ${descriptor}`, err);
            obj = null;
        }
        return obj;
    }

    // returns a callback function that captures the lexical scope (parameter value and private members)
    private handleConfirmation(pid: string, registration: boolean): () => void {
        let operation: string;
        let path: string;
        if (registration) {
            operation = 'registered';
            path = '/confirm-registration';
        } else {
            operation = 'unregistered';
            path = '/confirm-deregistration';
        }
        return () => {
            Logger.log(`Transaction confirmed: source with PID ${pid} successfully ${operation}`);
            const fullPath = '/members/' + pid + path;
            try {
                this.govProxy.put(fullPath).then((response) => {
                    if (response.status === 202) {
                        Logger.debug(`Call to GOV callback service ${fullPath} successful`);
                    } else {
                        // the call to the GOV callback service failed: log the response
                        Logger.error(`Call to GOV callback service ${fullPath} FAILED, status code is ${response.status}`);
                    }
                }).catch((err) => {
                    // we failed to call the GOV callback service: log and swallow the error
                    Logger.error(`Call to GOV callback service ${fullPath} FAILED - ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
                });
            } catch (err) {
                // we failed to call the GOV callback service: log and swallow the error
                Logger.error(`Error setting up the call to the GOV callback service ${fullPath} - ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
        };
    }
}
