import { Body, Controller, Get, HttpCode, HttpException, HttpStatus, Logger, Param, Post, UseGuards } from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiResponse, ApiTags } from '@nestjs/swagger';
import { AuthGuard } from 'src/auth/auth-guard';
import { StatsService } from './stats-service';
import { CatalogueStats } from 'src/dtos/catalogue-stats';
import { HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_500 } from 'src/constants';
import { BlockchainStats } from 'src/dtos/blockchain-stats';
import { RolesGuard } from 'src/auth/roles-guard';
import { Roles } from 'src/auth/roles-decorator';
import { ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN } from 'src/auth/auth-constants';
import axios, { AxiosInstance } from 'axios';
import { ConfigService } from '@nestjs/config';
import { User } from 'src/auth/user-decorator';
import { UserInfo } from 'src/auth/user-info';

@ApiTags('Open API: Catalogue & Blockchain Stats')
@Controller('/api/v1.0')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class StatsController {

    private readonly internalProxy: AxiosInstance = null;

    constructor(
        private readonly config: ConfigService,
        private readonly statsService: StatsService
    ) {
        this.internalProxy = axios.create({
            baseURL: this.config.get<string>('INTERNAL_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
    }
    
    @Get('/stats')
    @HttpCode(200)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieves some basic statistics of the FAME Marketplace Catalogue',
        description: `This operation provides a very generic overview of the current state of the FAME Marketplace Catalogue.`
    })
    @ApiResponse({
        status: 200, // OK
        description: `Returns an object containing the desired catalogue stats`,
        type: CatalogueStats
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async getCatalogueStats(): Promise<CatalogueStats> {
        return await this.statsService.getCatalogueStats();
    }

    @Get('/blockchain/stats')
    @HttpCode(200)  
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieves the health status of the FAME Blockchain network',       
        description: `This operation provides the current health status of the connection to the FAME Blockchain network.
  This operation is only available to administrators.`
    })
    @ApiResponse({
        status: 200, // OK
        description: `Returns an object containing the desired blockchain health status`,
        type: BlockchainStats
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })    
    getBlockchainStats(): BlockchainStats {
        return this.statsService.getBlockchainStats();
    }

    @Post('/blockchain/reconnect')
    @HttpCode(204)  
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Reconnects to the FAME Blockchain network',
        description: `This operation forces the PT service to re-establish its connection with the FAME Blockchain network.
  This operation is only available to administrators.`
    })
    @ApiResponse({ 
        status: 204, // No Content
        description: 'Reconnection initiated'
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })    
    async forceBlockchainReconnect(@User() user: UserInfo): Promise<void> {
        Logger.log('Blockchain connection reset requested');
        Logger.log('Calling user: ' + JSON.stringify(user));
        await this.statsService.forceBlockchainReconnect();
    }

    @Get('/blockchain/stats-internal')
    @HttpCode(200)  
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieves the health status of the FAME Blockchain network, from the perspoective of the INTERNAL PT service',       
        description: `This operation provides the current health status of the connection to the FAME Blockchain network, from the perspoective of the INTERNAL PT service.
  This operation is only available to administrators.`
    })
    @ApiResponse({
        status: 200, // OK
        description: `Returns an object containing the desired blockchain health status`,
        type: BlockchainStats
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })    
    async getBlockchainStatsInternal(): Promise<BlockchainStats> {
        try {
            const { data } = await this.internalProxy.get('/blockchain/stats');
            return data;
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Post('/blockchain/reconnect-internal')
    @HttpCode(204)  
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Reconnects the INTERNAL PT service to the FAME Blockchain network',
        description: `This operation forces the INTERNAL PT service to re-establish its connection with the FAME Blockchain network.
  This operation is only available to administrators.`
    })
    @ApiResponse({ 
        status: 204, // No Content
        description: 'Reconnection initiated'
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })    
    async forceBlockchainReconnectInternal(@User() user: UserInfo): Promise<void> {
        Logger.log('Blockchain connection reset requested for INTERNAL service');
        Logger.log('Calling user: ' + JSON.stringify(user));
        Logger.log('Call forwarded to INTERNAL service endpoint');
        try {
            await this.internalProxy.post('/blockchain/reconnect', {});
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }
}