import { ConfigService } from '@nestjs/config';
import { UserInfo } from '../auth/user-info';
import { ROLE_ADMIN, ROLE_OPERATOR } from '../auth/auth-constants';
import { ForbiddenException, BadRequestException, InternalServerErrorException, BadGatewayException, NotFoundException } from '@nestjs/common';
import { FdacService } from 'src/integration/fdac-service';
import { ApmService } from 'src/integration/apm-service';
import { GovService } from 'src/integration/gov-service';

const HTTP_EXCEPTION_SEP = '§§';
const HTTP_EXCEPTION_SEP_LEN = HTTP_EXCEPTION_SEP.length;

export enum ErrorCondition {
    AUTH,
    INPUT,
    NOTFOUND,
    EXTERNAL,
    INTERNAL,
    UNKNOWN,
}

export function isInternalApi(configService: ConfigService): boolean {
    return isTrue(configService, 'INTERNAL_API');
}

export function isAuthDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_AUTH');
}

export function isRoleCheckDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_ROLES');
}

export function isFdacDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_FDAC');
}

export function isApmDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_APM');
}

export function isXcheckDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_OFFERING_XCHECK');
}

export function isTmDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_TM');
}

export function isRqueueDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_RQUEUE');
}

export function getISOTimeFromString(seconds: string): string {
    return getISOTime(safeParseInt(seconds));
}

export function getISOTime(seconds: number): string {
    return seconds > 0 ? new Date(seconds * 1000).toISOString() : ''; // normalize non-positive values to empty string
}

export function safeParseInt(num: string) {
    return parseInt(num) || 0; // normalize empty or non-numerical value, which return 'NaN', to 0
}

export function isPrivilegedUser(user: UserInfo): boolean {
    return (user && (user.role === ROLE_ADMIN || user.role === ROLE_OPERATOR));
}

export function isAdminUser(user: UserInfo): boolean {
    return (user && user.role === ROLE_ADMIN);
}

function isTrue(conf: ConfigService, key: string): boolean {
    let val: string = conf.get<string>(key);
    return val.trim() === 'true';
}

// ensures that the private key, if set, always starts with the hexadecimal prefix
export function getPrivateKeyPT(config: ConfigService): string {
    const rawPrivateKey = config.get<string>('PRIVATE_KEY_PT');
    if (rawPrivateKey) {
        return rawPrivateKey.startsWith('0x') ? rawPrivateKey : '0x' + rawPrivateKey;
    } else {
        return rawPrivateKey;
    }
}

// ensures that the private key, if set, always starts with the hexadecimal prefix
export function getPrivateKeyPermission(config: ConfigService): string {
    const rawPrivateKey = config.get<string>('PRIVATE_KEY_PERMISSION');
    if (rawPrivateKey) {
        return rawPrivateKey.startsWith('0x') ? rawPrivateKey : '0x' + rawPrivateKey;
    } else {
        return rawPrivateKey;
    }
}

export function throwExceptionWithErrorCondition(type: ErrorCondition, desc: string) {
    throw new Error(type + HTTP_EXCEPTION_SEP + desc);
}

export function getErrorCondition(err: Error): ErrorCondition {
    const msg = err.message;
    if (!msg || msg.trim().length == 0) {
        return ErrorCondition.UNKNOWN;
    }
    const idx = msg.indexOf(HTTP_EXCEPTION_SEP);
    if (idx >= 0) {
        const typeString = msg.substring(0, idx);
        const typeNumber = Number(typeString);
        // Check if the parsed number is a valid enum member.
        if (!isNaN(typeNumber) && ErrorCondition[typeNumber] !== undefined) {
            return typeNumber as ErrorCondition;
        }
    }
    return ErrorCondition.UNKNOWN;
}

export function getErrorConditionDescription(err: Error): string {
    const msg = err.message;
    if (!msg || msg.trim().length == 0) {
        return msg;
    }
    const idx = msg.indexOf(HTTP_EXCEPTION_SEP);
    if (idx >= 0) {
        return msg.substring(idx + HTTP_EXCEPTION_SEP_LEN);
    } else {
        return msg;
    }
}

export function translateToHttpException(err: Error) {
    const condition: ErrorCondition = getErrorCondition(err);
    const message: string = getErrorConditionDescription(err);
    switch (condition) {
        case ErrorCondition.AUTH:
            throw new ForbiddenException(message);
        case ErrorCondition.INPUT:
            throw new BadRequestException(message);
        case ErrorCondition.NOTFOUND:
            throw new NotFoundException(message);
        case ErrorCondition.INTERNAL:
            throw new InternalServerErrorException(message);
        case ErrorCondition.EXTERNAL:
            throw new BadGatewayException(message);
        default:
            throw new InternalServerErrorException(message);
    }
}

export async function checkPublishingRules(
    fdacService: FdacService,
    apmService: ApmService,
    govService: GovService | null,
    aid: string,
    user: UserInfo,
    skipAuth: boolean,
    tid?: string
) : Promise<any> {
    const { Logger } = require('@nestjs/common');
    const { FAME_PFX } = require('../constants');
    let target: any = null;
    try {
        target = await fdacService.getAssetDcatDescription(aid);
    } catch (err) {
        Logger.error(`Error calling the FDAC module for checks on asset ${aid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
        throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Error communicating with the FDAC module');
    }

    if (target) {
        if (!await apmService.checkEditable(aid, user)) {
            Logger.warn(`AID ${aid} is not a valid publishing target according to APM policies`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, `Operation rejected: target asset was not found in the catalogue`);
        }
    } else {
        Logger.warn(`AID ${aid} is not a valid target, no entry found in FDAC`);
        throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, `Operation rejected: target asset was not found in the catalogue`);
    }
    
    if ('dcterms_creator' in target && target.dcterms_creator) {
        let rawCreator = target.dcterms_creator;

        let pid: string | null = null;
        if (rawCreator.startsWith(FAME_PFX)) {
            pid = rawCreator.slice(FAME_PFX.length);
        } else {
            Logger.warn(`The dcterms_creator attribute "${rawCreator}" does not start with the expected prefix "${FAME_PFX}".`);
        }

        if (pid) {
            if (skipAuth) {
                Logger.debug('AAI is disabled, skipping asset publisher and beneficiary account verification');
            } else {
                const affiliation = user.affiliation;
                if (pid !== affiliation) {
                    Logger.warn(`User ${user.uid} is affiliated to PID ${affiliation}, which does not match asset publisher PID ${pid}`);
                    throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'User is not affiliated to the publisher of the target asset');
                }
            }

            if (tid && govService) {
                if (!await govService.checkAccountAffiliation(tid, pid)) {
                    Logger.warn(`Beneficiary account ${tid} is not a valid account for asset publisher with PID ${pid}`);
                    throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Beneficiary account is not valid for the target asset');
                }
            }
        } else {
            Logger.warn(`The 'pid' value found in the catalogue entry for asset ${aid} is not in the expected format: skipping affiliation and beneficiary account verification.`);
            throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Cannot determine the publisher of the target asset');
        }
    } else {
        Logger.warn(`No 'pid' attribute found in the catalogue entry for asset ${aid}: skipping affiliation and beneficiary account verification.`);
        throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Cannot determine the publisher of the target asset');
    }

    return target;
}

