/**
 * Secure Markdown to HTML converter that only allows safe formatting
 * Allows: **bold**, *italic*, # headings, lists (- * 1.)
 * Blocks: links, images, code blocks, HTML tags, and other potentially dangerous content
 * Uses the exact same logic as the frontend for perfect alignment
 */
export class MarkdownUtils {

    /**
     * Convert safe Markdown to HTML
     * @param markdown The markdown text to convert
     * @returns HTML string with only safe formatting
     * Uses the exact same logic as the frontend for perfect alignment
     */
    public static markdownToHtml(markdown: string): string {
        if (!markdown || typeof markdown !== 'string') {
            return '';
        }

        try {
            let html = markdown;

            // Convert headers (only levels 1 and 2 allowed) - order matters: ## before #
            // Preserve original spacing for lossless conversion
            html = html.replace(/^(##)(\s+)(.+)$/gm, (match, hashes, spaces, content) => {
                return `<h2 data-hashes="${hashes}" data-spaces="${spaces}">${content}</h2>`;
            });
            html = html.replace(/^(#)(\s+)(.+)$/gm, (match, hashes, spaces, content) => {
                return `<h1 data-hashes="${hashes}" data-spaces="${spaces}">${content}</h1>`;
            });

            // Convert bold text
            html = html.replace(/\*\*([^*\n]+)\*\*/g, '<strong>$1</strong>');

            // Convert italic text (make sure it's not part of bold)
            html = html.replace(/(?<!\*)\*([^*\n]+)\*(?!\*)/g, '<em>$1</em>');

            // Convert unordered lists (preserving original formatting for lossless conversion)
            html = html.replace(/^(\s*[-*+]\s+.+(?:\r?\n\s*[-*+]\s+.*)*)/gm, (match) => {
                // Split by line and process each list item
                const lines = match.split(/\r?\n/);
                const listItems = lines
                    .map(line => {
                        // Capture original formatting details
                        const lineMatch = line.match(/^(\s*)([-*+])(\s+)(.*)$/);
                        if (lineMatch) {
                            const [, indent, marker, spaces, content] = lineMatch;
                            return {
                                content: content.trim(),
                                indent: indent,
                                marker: marker,
                                spaces: spaces
                            };
                        }
                        return null;
                    })
                    .filter(item => item !== null && item.content.length > 0) // Remove empty items
                    .map(item => `<li data-marker="${item.marker}" data-indent="${item.indent}" data-spaces="${item.spaces}">${item.content}</li>`)
                    .join('');
                return `<ul>${listItems}</ul>`;
            });

            // Convert ordered lists (preserving original formatting for lossless conversion)
            html = html.replace(/^(\s*\d+\.\s+.+(?:\r?\n\s*\d+\.\s+.*)*)/gm, (match) => {
                // Split by line and process each list item
                const lines = match.split(/\r?\n/);
                const listItems = lines
                    .map(line => {
                        // Capture original formatting details
                        const lineMatch = line.match(/^(\s*)(\d+\.)(\s+)(.*)$/);
                        if (lineMatch) {
                            const [, indent, marker, spaces, content] = lineMatch;
                            return {
                                content: content.trim(),
                                indent: indent,
                                marker: marker,
                                spaces: spaces
                            };
                        }
                        return null;
                    })
                    .filter(item => item !== null && item.content.length > 0) // Remove empty items
                    .map(item => `<li data-marker="${item.marker}" data-indent="${item.indent}" data-spaces="${item.spaces}">${item.content}</li>`)
                    .join('');
                return `<ol>${listItems}</ol>`;
            });

            // Handle line breaks intelligently:
            // - Don't add <br> immediately after block elements (headings, lists) unless there are extra newlines
            // - Convert regular text newlines to <br>

            // Step 1: Handle multiple newlines after block elements (user wants extra spacing)
            // Convert patterns like: </h1>\n\n or </ul>\n\n\n to: </h1>\n<br>\n or </ul>\n<br>\n<br>\n
            // This preserves BOTH the original newlines AND adds visual <br> tags
            html = html.replace(/(<\/(?:h[1-6]|ul|ol)>)\n(\n+)/g, (match, closingTag, extraNewlines) => {
                const numExtraLines = extraNewlines.length;
                return closingTag + '\n' + '<br>\n'.repeat(numExtraLines);
            });

            // Step 2: Keep single newlines after block elements for lossless conversion
            // They don't affect HTML rendering but allow perfect markdown reconstruction

            // Step 3: Handle double newlines for paragraph breaks (but not after block elements)
            html = html.replace(/\n\s*\n/g, '</p><p>');

            // Step 4: Convert remaining single newlines to <br> (these are in regular text)
            // BUT preserve structural newlines after block elements
            // Use a temporary placeholder to protect newlines after block elements
            html = html.replace(/(<\/(?:h[1-6]|ul|ol)>)\n/g, '$1STRUCTURAL_NEWLINE');

            // Convert all other newlines to <br>
            html = html.replace(/\n/g, '<br>');

            // Restore structural newlines
            html = html.replace(/STRUCTURAL_NEWLINE/g, '\n');

            // Wrap in paragraphs if not already wrapped in other tags
            if (!html.match(/^<[h1-6ul ol]/)) {
                html = `<p>${html}</p>`;
            }

            // Clean up empty paragraphs and fix paragraph nesting
            html = html.replace(/<p><\/p>/g, '');
            html = html.replace(/<p>(<[h1-6ul ol][^>]*>)/g, '$1');
            html = html.replace(/(<\/[h1-6ul ol]>)<\/p>/g, '$1');

            // Additional security: strip any remaining dangerous patterns
            return this.sanitizeHtml(html);

        } catch (error) {
            // If markdown parsing fails, return the original text
            console.warn('Markdown parsing failed:', error);
            return this.escapeHtml(markdown);
        }
    }

    /**
     * Additional HTML sanitization to remove any dangerous content
     */
    private static sanitizeHtml(html: string): string {
        return html
            // Remove script tags
            .replace(/<script[^>]*>.*?<\/script>/gi, '')
            // Remove style tags
            .replace(/<style[^>]*>.*?<\/style>/gi, '')
            // Remove iframe tags
            .replace(/<iframe[^>]*>.*?<\/iframe>/gi, '')
            // Remove object tags
            .replace(/<object[^>]*>.*?<\/object>/gi, '')
            // Remove embed tags
            .replace(/<embed[^>]*>/gi, '')
            // Remove form tags
            .replace(/<form[^>]*>.*?<\/form>/gi, '')
            // Remove input tags
            .replace(/<input[^>]*>/gi, '')
            // Remove onclick and other event handlers
            .replace(/\son\w+="[^"]*"/gi, '')
            .replace(/\son\w+='[^']*'/gi, '')
            // Remove javascript: links
            .replace(/javascript:[^"']*/gi, '');
    }

    /**
     * Escape HTML special characters
     */
    private static escapeHtml(text: string): string {
        const div = { innerHTML: '' } as any;
        div.textContent = text;
        return div.innerHTML || text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');
    }

    /**
     * Check if text contains any Markdown formatting
     */
    public static hasMarkdownFormatting(text: string): boolean {
        if (!text) return false;

        // Check for common markdown patterns (consistent with frontend)
        const markdownPatterns = [
            /\*\*[^*\r\n]+\*\*/,         // **bold**
            /(?<!\*)\*[^*\r\n]+\*(?!\*)/,  // *italic* (not part of **)
            /^#{1,2}\s+.+$/gm,           // # headings (only levels 1 and 2)
            /^\s*[-*+]\s+.+$/gm,         // unordered lists (global multiline)
            /^\s*\d+\.\s+.+$/gm,         // ordered lists (global multiline)
            /\r?\n\s*\r?\n/,             // paragraph breaks (double newlines with optional \r)
            /\r?\n/,                     // single newlines (line breaks)
        ];

        return markdownPatterns.some(pattern => pattern.test(text));
    }
}