#!/usr/bin/env node

/**
 * Verify the Tracing contract has the expected methods
 * Usage: node verify-contract.js
 */

const { Web3 } = require('web3');
const fs = require('fs');
const path = require('path');

// Configuration from environment or defaults
const CHAIN_URL = process.env.CHAIN_URL || 'ws://rpc-node:8546/websocket';
const CONTRACT_ADDRESS = process.env.CONTRACT_TRACING_ADDR || '0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce';
const ABI_PATH = process.env.CONTRACT_TRACING_ABI || 'integration/Tracing.abi';

async function verifyContract() {
    console.log(`Verifying Tracing Contract`);
    console.log(`Chain URL: ${CHAIN_URL}`);
    console.log(`Contract Address: ${CONTRACT_ADDRESS}`);
    console.log('');

    let web3;
    try {
        // Connect to blockchain
        console.log('Connecting to blockchain...');
        web3 = new Web3(CHAIN_URL);

        // Load contract ABI
        const abiAbsPath = path.join(__dirname, 'src', ABI_PATH);
        const abi = JSON.parse(fs.readFileSync(abiAbsPath, 'utf8'));

        console.log('Contract ABI loaded, methods:');
        const methods = abi.filter(item => item.type === 'function');
        methods.forEach(m => {
            console.log(`  - ${m.name}(${m.inputs.map(i => i.type).join(', ')})`);
        });

        const events = abi.filter(item => item.type === 'event');
        console.log('\nContract Events:');
        events.forEach(e => {
            console.log(`  - ${e.name}(${e.inputs.map(i => i.type + ' ' + i.name).join(', ')})`);
        });

        // Create contract instance
        const contract = new web3.eth.Contract(abi, CONTRACT_ADDRESS);

        // Try to call a read-only method to verify contract exists
        console.log('\nVerifying contract is deployed at this address...');

        try {
            // Get bytecode at the address
            const code = await web3.eth.getCode(CONTRACT_ADDRESS);

            if (code === '0x' || code === '0x0') {
                console.log('❌ ERROR: No contract deployed at this address!');
                console.log('The address contains no bytecode.');
                process.exit(1);
            } else {
                console.log(`✓ Contract exists at address (bytecode length: ${code.length} bytes)`);
            }

            // Try to call getAsset with a dummy AID
            console.log('\nTesting getAsset method with dummy AID...');
            const testAid = 'test123';
            const result = await contract.methods.getAsset(testAid).call();
            console.log(`✓ getAsset() call successful`);
            console.log(`  Result: ${JSON.stringify(result, null, 2)}`);

        } catch (error) {
            console.log(`⚠ Warning: Could not verify contract methods: ${error.message}`);
        }

        console.log('\n=== Contract Verification Complete ===');
        process.exit(0);

    } catch (error) {
        console.error('Error:', error.message);
        if (error.stack) {
            console.error('Stack:', error.stack);
        }
        process.exit(1);
    } finally {
        // Cleanup connection
        if (web3 && web3.provider) {
            try {
                await web3.provider.disconnect();
            } catch (e) {
                // Ignore cleanup errors
            }
        }
    }
}

verifyContract();
