# Project FAME - Trading and Monetisation Applications Monorepo

## Data Provider Integration

Data Provider Integration is described in the [DATA_PROVIDER_INTEGRATION.md](docs/DATA_PROVIDER_INTEGRATION.md) file.

## Overview

This repository contains the source code for the Trading and Monetisation applications for the FAME project. The project is a part of the European Union's Horizon 2020 research and innovation programme.

### Applications Description

There are two main backand applications in this repository:

1. **Trading & Monetization Private Application**:

   - Provides **backend-to-backend** interaction with smart contracts.
   - Manages **offerings, governance transactions, and internal data access.**
   - **Not exposed to public users** and enforces strict access control.

2. **Trading & Monetization Public Application**:

   - Provides **public APIs** for external users and platforms.
   - Supports **Governance, Payments, Trading History**, and **Data Access Transactions**.
   - Securely **exposes contract interactions while enforcing authentication and rate limits**.

3. **Fame Data Access Processor (Reference)**
   - Reference microservice for **validating dataset access** based on on-chain tokens.
   - Designed to run on a **data provider's infrastructure**.
   - See [apps/data-gate/README.md](apps/data-gate/README.md) for detailed setup instructions, endpoints, and architecture.

Applications reuse common libraries:

1. **Contracts**: Contains smart contracts for the FAME project.
2. **API-Shared**: Contains shared code for the private and public applications.

![Swagger](docs/images/app-dependency-tree.png 'App Dependency Tree')

Additionally, the repository contains

3.  **Trading & Monetization UI Demo application** \*_tm-ui-demo_ application is a simple Angular frontend app that demonstrates how to interact with the Trading and Monetisation backend, especially how to sign unsigned transactions.

It's dependencies are not added to global package.json in order to prevent of unneccessary angular dependencies in the main docker images (we are using integrated nx monorepo).
To install the dependencies for the `tm-ui-demo` application, you need to be in the `apps/tm-ui-demo` folder and run `npm install`.

For more information, please refer to the `tm-ui-demo` [README file](apps/tm-ui-demo/README.md).

## Development Getting Started

### Installation

1. To install all the necessary packages, run the following in terminal:

   ```sh
   npm install
   ```

2. Set up `.env` variables. Copy the `.env.example` file to `.env` and fill in the necessary variables.

You can change the used network by the server by setting up one varia

```
# Besu RPC address and port
BESU_RPC=162.55.94.15:8545

# Infura API Key
INFURA_API=

# Private key for managing contracts by server
ADMIN_PRIVATE_KEY=

# Endpoint that should be called for confirming offering
PT_CONFIRM_OFFERING_ENDPOINT=

# Which blockchain should be used by the server: besu | sepolia | localhost | hardhat
HARDHAT_NETWORK=localhost

```

### Setting up the network

3. If you set up `HARDHAT_NETWORK=localhost`, you also need to run local blockchain by running:

   ```sh
   npm run contracts:node:start

   ```

4. Set up Wallet.

   1. For BESU, it's recommended to add Besu network wallet in Metamask with the following configuration:

   ```
   Name: Besu
   RPC: http://162.55.94.15:8545/
   Chain ID: 6666
   Currency: ENG
   Block Explorer: http://162.55.94.15:25000/explorer/explorer
   ```

5. For local Hardhat Development, use the following configuration:

````
Name: Localhost 8545
RPC: http://localhost:8545
 Chain ID: 31337
 Currency: GO

```

5. You need to run `npm run contracts:compile` in order to compile the contracts before running the application.

```sh
npm run contracts:compile
````

6. To deploy contracts on the network, run:
   `sh
npm run contracts:deploy
`
   Deployed contracts will be saved in `contract-addresses.json` file and `diamonds-info.json` file.

7. After contractt deployment, you need to initialize the contracts storage by running:

   ```sh
   npm run contracts:initialize
   ```

8. Depoloyed contract `TradingManagement` can be invested by running:

   ```sh
   npm run contracts:trading-management:loupe
   ```

9. If all the contracts are deployed, you can run Server Apps:

- To run the private application, run:

  ```sh
  npm run tm-private:start
  ```

- To run the public application, run:

  ```sh
  npm run tm-public:start
  ```

10. If you want to use frontend demo application for testing or development purposes, please refer to the [tm-ui-demo README file](apps/tm-ui-demo/README.md) for more information.
11. In the monorepo there is also a proof of concept data provider application called `data-provider-poc`. Please refer to the [data-provider-poc README file](apps/data-provider-poc/README.md) for more information.
12. There is also a reference implementation of data access validator service called `data-gate`. Please refer to the [data-gate README file](apps/data-gate/README.md) for more information.

### Running the Application in Development Mode

All the important scripts are defined in the `package.json` file. They are divided into several categories:

- **tm-private**: Scripts for the private application.
- **tm-public**: Scripts for the public application.
- **contracts**: Scripts for the smart contracts.
- **all**: Scripts that run on all the source code, like `all:lint`, `all:clean`, etc.

## Smart Contract Unit Tests

To run the unit tests for the smart contracts, run:

```sh
npm run contracts:test

```

## App e2e tests

To run the e2e tests for the Trading and Monetisation applications, run:

```sh
npm run contracts:node:start:test
npm run contracts:deploy:test
npm run contracts:initialize:test
npm run tm-tests:all:e2e-tests

```

## Good practices

1. All features should be developed in a separate branch and merged to the `main` branch as a `merge request`.
2. All new functionalities should be covered by unit tests, especially smart contracts.
3. New features should be added to e2e tests.
4. All new endpoints should be documented in the Swagger documentation:
   - all parameters should be described
   - all responses should be described
   - all possible errors should be described
   - all possible status codes should be described
5. Code should be formatted using `prettier` and `eslint` rules.

## Docker Image Preparation and Publishing

This project is configured to use Docker for creating and managing the application's container. The following npm scripts are provided to simplify Docker operations such as building, pushing, and running the Docker image locally.

> ### CI/CD Note
>
> The default `Dockerfile` (without any suffixes) is configured specifically for **CI/CD pipelines and production builds**.  
> Do **not modify** it for local development — use `Dockerfile.private.local` instead.

### Building the Docker Image Locally

Dockerfile.private.local

Dockerfile variant intended for local builds that may include custom settings, test dependencies, or developer-specific logic.
While it can be used publicly, it's optimized for developer use cases rather than production pipelines.
To build and run the image you must provide all required environmental variables in .env file in root directory.

### Building the Docker Image

To build the Docker image for this project, run:

```
npm run tm-public:docker:build
npm run tm-private:docker:build

```

This commands run
Dockerfile.private.local → Used for local development with private dependencies/configuration.
This command executes a Docker build operation with the tag `harbor.gftinnovation.eu/fame/framework/tm/{private|public}:manual`, using the Dockerfile located at `devops` folder. Ensure you are in the project root directory when running this command.

### Pushing the Docker Image to a Registry

Before pushing the Docker image, you must be logged in to the Docker registry. To push the built image to the `harbor.gftinnovation.eu` registry, run:

```
npm run tm-public:docker:push
npm run tm-private:docker:push
```

This script will first log you into the `harbor.gftinnovation.eu` registry (you'll be prompted for your credentials), and then push the `harbor.gftinnovation.eu/fame/framework/tm/{private|public}:latest` image to the registry.

> Latest docker images will be published here:
> PUBLIC APP: `[harbor.gftinnovation.eu/fame/framework/tm/public](https://harbor.gftinnovation.eu/harbor/projects/40/repositories/framework%2Ftm%2Fpublic)`
> PRIVATE APP: `[harbor.gftinnovation.eu/fame/framework/tm/private](https://harbor.gftinnovation.eu/harbor/projects/40/repositories/framework%2Ftm%2Fprivate)`

### Running the Docker Image

To run the Docker image locally, use:

```
npm run tm-public:docker:run
npm run tm-private:docker:run
```

This command will start a container named `fame-tm-public|private` from the `harbor.infinitech-h2020.eu/fame/tm/{private|public}:manual` image. The application inside the container will be accessible through port 3000 (private app) or 3001 (pubic app) on your local machine.

## Usage

### Deploying Smart Contracts (and upgrading)

To deploy the necessary contracts:

1. Check the `.env` file for the correct network and private key.
2. Run the following command:

```
npm run contracts:deploy
npm run contracts:initialize
```

### Running Development Server with Swagger API

To run:

```
npm run tm-private:start
npm run tm-public:start
```

After that to access the swagger documentation and interactions, use this URL:
http://URL/swagger

## Codebase Structure

The repository is organized into several folders and files, each serving a specific purpose in the overall architecture.

- **apps**: Contains the source code for the Trading and Monetisation applications.
- **libs**: Contains shared code for the private and public applications.
- **devops**: Contains Dockerfile for building the Docker image.
- **docs**: Contains documentation and images.
- **deploy**: Contains scripts for Kubernetes deployment.

### Contracts Folder

The `contracts` folder contains the smart contracts for the FAME project. The contracts are written in Solidity and are used to manage the data access tokens, payment tokens, escrow, governance, and bourse functionalities.

Placement: `libs/contracts/src/contracts`

- **OfferingToken.sol**: ERC-721 token contract for offering tokens.
- **BaseDataAccess.sol**: Base class for data access contracts, handling shared code.
- **DataAccessPAYG.sol, DataAccessPAYU.sol, DataAccessSubscription.sol**: Derived contracts handling different data access methods.
- **Escrow.sol**: Manages escrow functionalities.
- **PaymentToken.sol**: Basic ERC-20 token contract.
- **Governance.sol**:Handles governance functionalities such as minting payment tokens, retrieving contract addresses, and managing diamond contracts. Manages governance functionalities using `PaymentToken.sol`.
- **Bourse.sol (Diamond Contract)**: Tracks trading history and executes transactions between Data Access and Payment Tokens.
- **diamond/Diamond.sol**: Diamond pattern implementation for upgradability with all the diamond facets needed.
- **TradingManagement.sol (Diamond Contract)**: Manages all trading functionalities and enables **upgradability via facets**.
- **TradingManagementExecutorFacet.sol**: Diamond Facet for the TradingManagement contract that contains the main trading functionalities (purchase methods).
- **TradingManagementStorageFacet**: Diamond Facet for the TradingManagement contract that contains the storage variables (references to other contracts).

### 3. ABIs Folder and deployed contract addresses

- To access the ABIs for the deployed contracts, check the `/libs/contracts/artifacts/libs/contracts/src/contracts` folder.
- To access the **deployed contract addresses** check the `/libs/contracts/contract-addresses.json` file.
- To access the **deployed diamond contract addresses** check the `/libs/contracts/diamonds-info.json` file.

## Calling Order to Execute API Calls for Testing / Development Purposes

The following is the order of API calls that should be executed in order to test the Trading and Monetisation applications.
Preconditions:

- The relevant blockchain network is running.
- The smart contracts are deployed on the network.
- Two API applications are running (private and public).
- The user has a wallet with ENG tokens.
- The user wallet is authorized to send transactions.
- The user has a trading account.

1. **Create Data Offering**

   - **API Call**: `POST /tm/v1.0/offerings`
   - **Purpose**: Create a offering token/data.
   - **Body for Subscription**:
     ```json
     {
       "assetid": "asub1",
       "oid": "sub1",
       "resource": "http://resource.jpg",
       "beneficiary": "0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a",
       "price": "10",
       "cap_duration": "2034-05-31T17:46:51.107Z",
       "cds_target": { "key": "value" },
       "cds_sl": { "key": "value" }
     }
     ```
   - **Body for PAYG (because capVolume) **:

     ```json
     {
       "assetid": "apayg1",
       "oid": "payg1",
       "resource": "http://resource.jpg",
       "beneficiary": "0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a",
       "price": "10",
       "cap_duration": "2034-05-31T17:46:51.107Z",
       "cap_volume": "2",
       "cds_target": { "key": "value" },
       "cds_sl": { "key": "value" }
     }
     ```

     - **Body for PAYU (because capDownloads is not empty) **:

     ```json
     {
       "assetid": "apayu1",
       "oid": "payu1",
       "resource": "http://resource.jpg",
       "beneficiary": "0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a",
       "price": "10",
       "cap_duration": "2034-05-31T17:46:51.107Z",
       "cap_downloads": "3",
       "cds_target": { "key": "value" },
       "cds_sl": { "key": "value" }
     }
     ```

2. **Mint Payment Tokens**

   - **API Call**: `POST /tm/v1.0/payment-tokens/mint`
   - **Purpose**: Mint payment tokens in order to buy data access token in next steps. This is done by the governance module.
   - **Params**: Trading account ID (address).
   - **Body**:
     ```json
     {
       "amount": "1000000000000000",
       "tradingAccount": "0xf39Fd6e51aad88F6F4ce6aB8827279cffFb92266"
     }
     ```

3. **Approve Payment Tokens**

   - **Purpose**: Consumers approve the `PaymentToken.sol` to spend minted before tokens in the `FameBourse` contract. In production systems, this should be done by the consumer's wallet using the GUI. For testing purposes, this can be done using the API call.
   - **Controller**: `Testing (deprecated - for test purposes only)`
   - **API Call**: `POST /test-only/payment/approve-operator-to-spend-payment-tokens`

4. **Generate unsigned transaction to purchase Data Access Token**

   - **API Call**: `POST /api/v1.0/submissions/order`
   - **Purpose**: Generate unsigned transaction to purchase a data access token. User needs to sign this transaction using Metamask in trading GUI dashboard and submit it.
   - **Params**: `oid` representing the offering token that was minted with that same `oid`.

5. **Purchase Data Access Token [TEST ONLY if 6 step is not implmented]**

   - **API Call**: `POST /test-only/data-access/submissions/order`
   - **Purpose**: Generate unsigned transaction to purchase a data access token. User needs to sign this transaction using Metamask in trading GUI dashboard and submit it.
   - **Params**: `oid` representing the offering token that was minted with that same `oid`.

6. **Check Clearance**
   - **API Call**: `GET /tm/v1.0/check-clearance`
   - **Purpose**: Check if the user has data access tokens.
   - **Params**: `assetId` representing the offering token that was minted with that same `oid`.

## API Endpoints

The system provides two distinct API layers:

![Swagger](docs/images/swagger-tm-private.png 'Swagger API Documentation')

1. **Private API**

   - Restricted to **internal FAME services**.
   - Manages contract deployments, governance actions, and offering creation.

![Swagger](docs/images/swagger-tm-public.png 'Swagger API Documentation')

2. **Public API**

- Exposes **Governance, Payments, Trading History**, and **Data Access Transactions**.
- Provides unsigned transactions for **users to sign & submit via their wallets**.

All endpoints are documented using **Swagger** and can be accessed at `/swagger`.

## Tests

E2E tests of tm-public app
[tm-public-e2e](apps/tm-public-e2e/README.md)

## Upgrading Smart Contracts

### Diamond Pattern Contract

Documentation on why we want to use upgradable contracts and Diamond Pattern can be found in [upgradability.md](docs/upgradability.md) file.

There is a hardhat task defined in `hardhat.config.ts` for upgrading diamond pattern contracts. To upgrade the contracts, you need to run the following command:

```bash
$ npx hardhat upgrade-diamond --diamond-name TradingManagement --facet-name TradingManagementExecutorFacet
```

This task performs the following operations:

1. Reads the existing diamond deployment information from `diamonds-info.json`
2. Deploys a new version of the specified facet contract
3. For `BourseFacet`, links required libraries (AVLTreeLib, TokenUtilsLib, BourseLib)
4. Removes function selectors from the old facet implementation
5. Adds function selectors from the new facet implementation
6. Performs the diamond cut operation to complete the upgrade
7. Updates the diamond deployment information with the new facet address

**Note**: For `BourseFacet` upgrades, the `supportsInterface(bytes4)` selector is filtered out to avoid conflicts.

Diamond contract structure based on TradingManagement contract can be represeted as below:

![Diamond](docs/diagrams/level2/TradingManagementExecutor%20Diamond%20-%20C4%20Level%202.svg 'Diamond Pattern')

### Transparent Proxy Contracts

Currently transparent proxy pattern is used in the following contracts: OfferingToken, PaymentToken, DataAccessPAYG, DataAccessPAYU, DataAccessSubscription.

For contracts that follow the Transparent Proxy Pattern, a specific upgrade process is required. There is a Hardhat task defined in `hardhat.config.ts` for upgrading these contracts.

Befoere upgrading the contracts, make sure that:

1. You set up the `.env` file with the correct network and private key.
2. You have compiled the contracts using `npm run contracts:compile`.
3. Run the following command:

```bash
$ npx hardhat upgrade-transparent-proxy --proxyaddress <YOUR_PROXY_ADDRESS> --contractname <NEW_CONTRACT_NAME>
```

proxyaddress: The address of the existing transparent proxy contract that you can find in `contract-addresses.json` file for the relevant network.
contractname: The name of the new implementation contract that you want to deploy and upgrade to, e.g., "OfferingToken".

This task performs the following operations:

1. Gets the contract factory for the new implementation contract
2. Uses OpenZeppelin's `upgrades.upgradeProxy()` function to upgrade the proxy
3. The proxy address remains the same while the implementation is updated
4. All existing contract state is preserved during the upgrade

**Parameters:**

- `proxyaddress`: The address of the existing transparent proxy contract
- `contractname`: The name of the new implementation contract (e.g., "PaymentTokenV2")

### Transfer All Ownership

For administrative purposes, there's a Hardhat task that transfers ownership of all deployed contracts to a new administrator address. This is useful for production deployments where ownership needs to be transferred to a multisig wallet or different admin account.

```bash
$ npx hardhat transfer-all-ownerships --newowner <NEW_OWNER_ADDRESS>
```

This task will transfer ownership of the following contracts:

- Governance Diamond
- TradingManagement Diamond
- OfferingToken
- DataAccessPayAsYouGo
- DataAccessPayAsYouUse
- DataAccessSubscription
- Escrow

The task reads deployed contract addresses from `contract-addresses.json` and only transfers ownership if the current signer is the current owner of each contract.

### Set Payment Token Permissions

This task configures the permissions contract address for payment tokens via the GovernanceFacet. This is typically used to set up access control for payment token operations.

```bash
$ npx hardhat set-payment-token-permissions --diamondaddress <DIAMOND_ADDRESS> --permissionsaddress <PERMISSIONS_CONTRACT_ADDRESS> --paymenttokenaddress <PAYMENT_TOKEN_ADDRESS>
```

Parameters:

- `diamondaddress`: The address of the deployed Diamond contract
- `permissionsaddress`: The address of the new permissions contract to set
- `paymenttokenaddress`: The address of the payment token contract (used for verification)

The task will:

1. Connect to the GovernanceFacet interface at the Diamond address
2. Call `setPaymentTokenPermissions()` with the FDE token symbol and permissions address
3. Verify the change was applied correctly by checking the PaymentToken contract

**Note**: The executing account must be the owner of the Diamond contract.

### Read Data Access Token URI

This task allows you to read the current metadata URI for any of the Data Access Token contracts (Subscription, Pay-As-You-Go, or Pay-As-You-Use).

```bash
$ npx hardhat read-uri --token-type <TOKEN_TYPE>
```

**Parameters:**

- `--token-type`: The type of token to read. Must be one of `SUB`, `PAYG`, or `PAYU`.

The script will:

1. Connect to the appropriate Data Access Token contract based on the `TOKEN_TYPE`.
2. Read and display the current URI for the token contract.

### Update Data Access Token URI

This task allows you to update the metadata URI for any of the Data Access Token contracts (Subscription, Pay-As-You-Go, or Pay-As-You-Use).

```bash
$ npx hardhat update-uri --token-type <TOKEN_TYPE> --new-uri <NEW_URI>
```

**Parameters:**

- `--token-type`: The type of token to update. Must be one of `SUB`, `PAYG`, or `PAYU`.
- `--new-uri`: The new URI to set for the token contract.
- `--network`: The network where the contract is deployed (e.g., localhost, besu).

The script will:

1. Connect to the appropriate Data Access Token contract based on the `TOKEN_TYPE`.
2. Call `setURI()` with the `NEW_URI`.
3. Verify that the URI was updated correctly.

**Note**: The executing account must be the owner of the contract.

## Open questions / TODOs

### Public Application

1. Create repository of token uri json references (IPFS replacement);

### Contracts

1. Refactor the contracts to use transparent proxy pattern or / and the diamond pattern for upgradability.
   Upgradability patters and options are described in the [upgradability.md](docs/upgradability.md) file.
   Requirements:
   - The contracts should be upgradable without losing the historical state, eg:
     - trade history should have always historical trades even if we upgrade BourseContract;
     - we should be able to migrate to future Digital Euro payment token implementation;
     - we should be able to change the governance logic, eg. change the voting mechanism without a need to redeploy the contract and lose the historical events;
     - we should be able to change Data Access Token logic or add other types of tokens without losing the historical states (who bought what and when);
   - Upgradability should be possible without losing the contract address;
   - The contracts should be modular and easy to maintain;
   - The contracts should use well known patterns and libraries;
2. Rething burrning token business logic with escrow and governance.
3. Add images from assets to the NFT contracts - set up proper tokenURI.
4. Make sure that token is representing the offer from dashboard (hash of the offer in json token uri?).
5. Add unit tests for the smart contracts for missing functionalities.

#### Data Access Token refactor

1. Remove all dependencies on other contracts.
2. Remove any specific logic from the contract.
3. Move all the specific logic to the Trading Manager contract.
4. Make the Data Access Token Contracts upgradeable.
5. Uri should point to some meaningful data with the hash of the offer + asset image.

#### Trading Manager refactor

1. Use the diamond pattern for the Trading Manager contract.
2. It should be only contract with all the dependencies on other contracts for trading.
3. Dependencies on other contracts should by dynamic and upgradable.

## Dashboard Application

1.  Create page with information about Data Access Tokens bought by the user.
2.  Improvements in the UI/UX.

## Support

In order to get support, please contact the project team:
Krzysztof Saja <krzysztof.saja.external@fujitsu.com>
Can Ozturan <ozturanc@trailblu.com> (Specifically, Governance and Bourse contract related questions)
Tomasz Awramski <tomasz.awramski@fujitsu.com>

## Authors and acknowledgment

Krzysztof Saja <krzysztof.saja.external@fujitsu.com>
Rufat Aliyev <rufat.aliyev.ftb@fujitsu.com>
Can Ozturan <ozturanc@trailblu.com>
Alper Sen <alper.sen@trailblu.com>
Tomasz Awramski <tomasz.awramski@fujitsu.com>

## Project status

In development
