⚠️ **Reference Implementation**  
This project is provided as a **reference implementation** of a FAME-compliant Data Access Validator.

# FAME Data Access Validator

The **data-gate** is an internal microservice responsible for validating Data Access Tokens and executing associated on-chain operations as part of the FAME Platform trust framework.

It is designed to be deployed by individual data providers who offer paid access to their datasets through blockchain-based tokens issued on the FAME marketplace.

The validator supports both on-chain and off-chain validation strategies and ensures that only valid, authorized tokens result in access to protected resources.

It is intended to run as an internal, secured component in the data provider's infrastructure — forming a key part of decentralized access control for FAME-compliant data services.

## Features

**Validate Data Access Tokens:**

- ECDSA signature verification
- Token expiration checks (on-chain via smart contracts or off-chain via FAME API)
- Blockchain interactions via ERC-1155 contracts for token ownership and clearance checks
- Supports token types: Subscription (SUB), Pay-as-you-go (PAYG), Pay-as-you-use (PAYU)
- Withdraws payment tokens on-chain for PAYG tokens using ERC-20 transferFrom

## Folder Overview

- `src/app.module.ts`: Application root module
- `src/main.ts`: Application entry point
- `src/shared-local/contracts/`: ABI definitions for smart contracts
- `src/shared-local/services/contract-factory.service.ts`: Contract instances setup
- `src/shared-local/utils/hash.ts`: Hashing helpers
- `src/config/validation.ts`: Environment schema validation
- `src/dto/`: Request DTOs
- `src/modules/access-evaluator/`: Access validation logic
- `src/modules/blockchain-gateway/`: Blockchain interaction layer
- `src/modules/redeem-data-access/`: Payment processing controller
- `.env`: Environment variable declarations

## Setup Instructions

### 1. Prerequisites

Ensure the following are installed:

- Node.js v20+
- npm v9+ (comes with Node.js)
- Docker (optional, for running locally in a container)

### 2. Install Dependencies

```bash
npm install
```

### 3. Setup Environment Variables

Create a `.env` file in the project root with the following content:

```bash
BLOCKCHAIN_RPC_URL=http://162.55.94.15:8545
VALIDATION_METHOD=ON_CHAIN # or OFF_CHAIN
TM_API=https://tm.fame-horizon.eu/api/v1.0
PROVIDER_PRIVATE_KEY=0xprovider-private-key
JWT=your-jwt-token
```

**Validation Methods:**

Set via `VALIDATION_METHOD` env var:

- `ON_CHAIN` → Uses smart contracts to verify expiration + clearance
- `OFF_CHAIN` → Relies on external APIs (Trading Management API)

⚠️ **Important:**
The `OFF_CHAIN` validation mode supports signature verification, clearance checks, and expiration validation based on off-chain data via the FAME API. `ON_CHAIN` mode provides higher reliability but slower response times.

### 4. Run the Application

```bash
npm run start
```

The service will be available on: `http://localhost:8888`

### 5. Run with Docker (Optional)

```bash
docker build --network=host -f devops/Dockerfile -t fame-data-gate .
docker run --env-file .env -p 8888:8888 fame-data-gate
```

## ABI & Contract Metadata Access

Smart contract ABIs and contract addresses must be accessible through the Trading Management API.

All contract-related metadata can be fetched from:

```
https://tm.fame-horizon.eu/api/v1.0/
```

### Used Endpoints

- **GET** `/contract-info/addresses`  
  Retrieves the latest deployed contract addresses.

The `ContractFactoryService` internally uses these addresses when initializing ethers.js contract instances. Ensure that:

- Contract addresses are up-to-date with deployed versions
- Trading Management API is accessible with valid JWT token
- Provider wallet has sufficient ETH for gas fees

## API Endpoints

### GET `/api/v1.0/validate/signature`

Verifies that a given ECDSA signature matches the expected Ethereum address.

**Request Body:**

```json
{
  "message": "123:0xf2d72bf781c11186553884574c01175c7d205822:SUB",
  "signature": "0x1234567890abcdef...",
  "expectedAddress": "0xf2d72bf781c11186553884574c01175c7d205822"
}
```

**Responses:**

- `200 OK` – Returns `true` or `false` indicating signature validity
- `400 Bad Request` – Invalid input format or signature
- `401 Unauthorized` – Signature verification failed

### GET `/api/v1.0/validate/access-token`

Validates user access to a dataset based on token ownership, expiration, and signature verification.

**Request Body:**

```json
{
  "tokenId": "123",
  "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "tokenType": "SUB",
  "signature": "0x1234567890abcdef..."
}
```

**Responses:**

- `200 OK` – Returns `true` if access is granted
- `401 Unauthorized` – Signature invalid, token expired, or no clearance
- `400 Bad Request` – Invalid request structure

### POST `/api/withdraw-payment-tokens-PAYG`

Processes payment token withdrawal for PAYG data access.

**Query Parameters:**

```json
{
  "customerAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "amount": "1000"
}
```

**Responses:**

- `200 OK` – Returns transaction receipt:
  ```json
  {
    "success": true,
    "transactionHash": "0xabcdef...",
    "blockNumber": 12345,
    "gasUsed": "21000"
  }
  ```
- `400 Bad Request` – Invalid parameters or insufficient balance
- `401 Unauthorized` – Transaction failed

## Token Types

| Type     | Description               | Validation                          | Payment     |
| -------- | ------------------------- | ----------------------------------- | ----------- |
| **SUB**  | Subscription-based access | Expiration + ownership              | Pre-paid    |
| **PAYG** | Pay per data volume       | Volume cap + expiration + ownership | Per-request |
| **PAYU** | Pay per usage session     | Expiration + ownership              | Per-session |

## Security Considerations

- **Private Key Protection** – Store provider private keys securely
- **API Authentication** – Secure JWT tokens for Trading Management API
- **Signature Verification** – All requests must include valid cryptographic signatures
- **Gas Management** – Monitor gas costs for on-chain operations

## Troubleshooting

| Issue                                   | Cause                              | Solution                                       |
| --------------------------------------- | ---------------------------------- | ---------------------------------------------- |
| `401 Access validation failed`          | Invalid signature or expired token | Verify signature generation and token validity |
| `500 Failed to fetch token information` | TM API unreachable                 | Check `TM_API` URL and `JWT` token             |
| `500 Contract interaction failed`       | Blockchain connectivity            | Verify `BLOCKCHAIN_RPC_URL` and network status |

---

⚠️ **Important**: Ensure the Trading Management API is accessible and your provider wallet has sufficient ETH for gas fees before deploying to production.
