import { ApiProperty } from '@nestjs/swagger';
import {
  IsEnum,
  IsEthereumAddress,
  IsNotEmpty,
  IsString,
  Matches,
} from 'class-validator';

// Token type enum
export enum TokenType {
  SUB = 'SUB',
  PAYG = 'PAYG',
  PAYU = 'PAYU',
}

export class ValidateRequestDto {
  @ApiProperty({
    description: 'Token ID to validate',
    example: 'token_123',
  })
  @IsString()
  @IsNotEmpty()
  tokenId!: string;

  @ApiProperty({
    description: 'Ethereum address of the token holder',
    example: '0xf2d72bf781c11186553884574c01175c7d205822',
  })
  @IsEthereumAddress()
  @IsNotEmpty()
  senderAddress!: string;

  @ApiProperty({
    enum: TokenType,
    description:
      'Token type: SUB (Subscription), PAYG (Pay As You Go), or PAYU (Pay As You Use)',
    example: TokenType.SUB,
  })
  @IsEnum(TokenType, {
    message: `tokenType must be one of: ${Object.values(TokenType).join(', ')}`,
  })
  @IsNotEmpty()
  tokenType!: TokenType;

  @ApiProperty({
    description: 'Cryptographic signature for authentication',
    example: '0x1234567890abcdef...',
  })
  @IsString()
  @IsNotEmpty()
  @Matches(/^0x[a-fA-F0-9]+$/, {
    message: 'signature must be a valid hex string starting with 0x',
  })
  signature!: string;
}
