import { Controller, HttpCode, HttpStatus, Get, Query } from '@nestjs/common';
import {
  ApiOperation,
  ApiTags,
  ApiUnauthorizedResponse,
  ApiBadRequestResponse,
  ApiOkResponse,
  ApiQuery,
} from '@nestjs/swagger';
import { AccessEvaluatorService } from './access-evaluator.service';
import { ValidateRequestDto } from '../../dto/validate-request.dto';
import { ValidateSignatureDto } from '../../dto/validate-signature.dto';

@ApiTags('Data Access Validator')
@Controller('api/v1.0/validate')
export class AccessEvaluatorController {
  constructor(private readonly accessEvaluator: AccessEvaluatorService) {}

  @Get('signature')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({
    summary: 'Verify ECDSA signature',
    description:
      'Verifies a cryptographic signature against an expected Ethereum address',
  })
  @ApiQuery({
    type: ValidateSignatureDto,
    description: 'Signature validation request',
  })
  @ApiOkResponse({
    description: 'Signature validation result',
    schema: { example: true },
  })
  @ApiBadRequestResponse({
    description: 'Invalid input format or signature',
    schema: {
      example: {
        statusCode: 400,
        message: 'Invalid Ethereum address format',
        error: 'Bad Request',
      },
    },
  })
  @ApiUnauthorizedResponse({
    description: 'Signature verification failed',
    schema: {
      example: {
        statusCode: 401,
        message: 'Invalid signature',
        error: 'Unauthorized',
      },
    },
  })
  validateSignature(@Query() dto: ValidateSignatureDto): boolean {
    return this.accessEvaluator.validateSignature(
      dto.message,
      dto.signature,
      dto.expectedAddress,
    );
  }

  @Get('access-token')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({
    summary: 'Validate data access token',
    description:
      'Validates signature, expiration, and ownership for data access',
  })
  @ApiQuery({
    type: ValidateRequestDto,
    description: 'Access validation request with token details and signature',
  })
  @ApiOkResponse({
    description: 'Access validation successful',
    schema: { example: true },
  })
  @ApiUnauthorizedResponse({
    description:
      'Access denied - invalid signature, expired token, or insufficient permissions',
    schema: {
      example: {
        statusCode: 401,
        message: 'Token expired or invalid',
        error: 'Unauthorized',
      },
    },
  })
  @ApiBadRequestResponse({
    description: 'Invalid request body or parameters',
    schema: {
      example: {
        statusCode: 400,
        message: ['tokenId should not be empty'],
        error: 'Bad Request',
      },
    },
  })
  async validateAccess(@Query() dto: ValidateRequestDto): Promise<boolean> {
    return await this.accessEvaluator.validateAccess(dto);
  }
}
