import { Injectable, Logger, UnauthorizedException } from '@nestjs/common';
import { HttpService } from '@nestjs/axios';
import { ConfigService } from '@nestjs/config';
import { firstValueFrom } from 'rxjs';
import { ethers } from 'ethers';
import { BlockchainGatewayService } from '../blockchain-gateway/blockchain-gateway.service';
import { ValidateRequestDto } from '../../dto/validate-request.dto';

type TokenType = 'SUB' | 'PAYG' | 'PAYU';

@Injectable()
export class AccessEvaluatorService {
  private readonly logger = new Logger(AccessEvaluatorService.name);
  private readonly validationMethod: string;

  constructor(
    private readonly blockchainGatewayService: BlockchainGatewayService,
    private readonly httpService: HttpService,
    private readonly configService: ConfigService,
  ) {
    this.validationMethod =
      this.configService.getOrThrow<string>('VALIDATION_METHOD');
    this.logger.log(`Validation method: ${this.validationMethod}`);
  }

  async getOidFromDataAccessTokenId(tokenId: string): Promise<string> {
    try {
      const oid =
        await this.blockchainGatewayService.getOidFromDataAccessTokenId(
          tokenId,
        );
      this.logger.debug('Retrieved OID', { tokenId, oid });
      return oid;
    } catch (error) {
      this.logger.error('Failed to get OID', { tokenId, error: error.message });
      throw error;
    }
  }

  async getVolumeInKB(tokenId: string): Promise<number> {
    try {
      const offerId = await this.getOidFromDataAccessTokenId(tokenId);
      const volumeCap =
        await this.blockchainGatewayService.getVolumeInKBPAYG(offerId);
      this.logger.debug('Retrieved volume', { tokenId, volumeCap });
      return volumeCap;
    } catch (error) {
      this.logger.error('Failed to get volume', {
        tokenId,
        error: error.message,
      });
      throw error;
    }
  }

  validateSignature(
    message: string,
    signature: string,
    expectedAddress: string,
  ): boolean {
    try {
      const recovered = ethers.utils.verifyMessage(message, signature);
      const isValid = recovered.toLowerCase() === expectedAddress.toLowerCase();
      this.logger.debug('Signature validation', { expectedAddress, isValid });
      return isValid;
    } catch (error) {
      this.logger.error('Signature validation failed', {
        expectedAddress,
        error: error.message,
      });
      throw new UnauthorizedException('Invalid signature');
    }
  }

  async checkClearance(
    tokenId: string,
    senderAddress: string,
    tokenType: TokenType,
  ): Promise<boolean> {
    if (this.validationMethod === 'ON_CHAIN') {
      try {
        return await this.blockchainGatewayService.checkClearance(
          tokenId,
          senderAddress,
          tokenType,
        );
      } catch (error) {
        this.logger.error('On-chain clearance failed', {
          tokenId,
          error: error.message,
        });
        return false;
      }
    }

    try {
      const response = await firstValueFrom(
        this.httpService.get(
          `${this.configService.getOrThrow('TM_API')}/data-access-info/check-clearance`,
          {
            params: { assetId: tokenId, addresses: senderAddress, tokenType },
            headers: {
              Authorization: `Bearer ${this.configService.getOrThrow('JWT')}`,
            },
          },
        ),
      );
      return response.data;
    } catch (error) {
      this.logger.error('Off-chain clearance failed', {
        tokenId,
        error: error.message,
      });
      return false;
    }
  }

  async getTokenExpirationByType(
    address: string,
    tokenId: string,
    tokenType: TokenType,
  ): Promise<number[]> {
    if (this.validationMethod === 'ON_CHAIN') {
      try {
        switch (tokenType) {
          case 'SUB':
            return await this.blockchainGatewayService.getSubscriptionExpiration(
              address,
              tokenId,
            );
          case 'PAYG':
            return await this.blockchainGatewayService.getPAYGExpiration(
              address,
              tokenId,
            );
          case 'PAYU':
            return await this.blockchainGatewayService.getPAYUExpiration(
              address,
              tokenId,
            );
          default:
            return [0];
        }
      } catch (error) {
        this.logger.warn(
          'On-chain expiration failed, falling back to off-chain',
        );
      }
    }

    // Off-chain fallback
    const baseUrl = this.configService.getOrThrow('TM_API');
    const endpoints = {
      SUB: `${baseUrl}/subscription-all-expirations`,
      PAYG: `${baseUrl}/PAYG-all-expirations`,
      PAYU: `${baseUrl}/PAYU-all-expirations`,
    };

    try {
      const response = await firstValueFrom(
        this.httpService.get(endpoints[tokenType], {
          params: { userAddress: address, oid: tokenId },
          headers: {
            Authorization: `Bearer ${this.configService.getOrThrow('JWT')}`,
          },
        }),
      );

      const data = response.data || 0;
      return Array.isArray(data) ? data : [data];
    } catch (error) {
      this.logger.error('Off-chain expiration failed', {
        tokenType,
        error: error.message,
      });
      return [0];
    }
  }

  async validateAccess(dto: ValidateRequestDto): Promise<boolean> {
    const { tokenId, senderAddress, tokenType, signature } = dto;
    this.logger.log('Validating access', { tokenId, senderAddress, tokenType });
    const nonce = new Date().toISOString().slice(0, 16);
    // Validate signature
    const message = `FAME Data Access Authentication Request

By signing this message, you confirm your identity and authorize access to the specified data resource.

Token ID: ${tokenId}
TokenType: ${tokenType}
Wallet Address: ${senderAddress}
Timestamp: ${nonce}`;

    const validSig = this.validateSignature(message, signature, senderAddress);
    if (!validSig) {
      throw new UnauthorizedException('Invalid signature');
    }

    // Check expiration
    const expirations = await this.getTokenExpirationByType(
      senderAddress,
      tokenId,
      tokenType,
    );
    const now = Math.floor(Date.now() / 1000);
    const validExpirations = expirations.some(
      (exp) => BigInt(exp.toString()) > BigInt(now),
    );

    if (!validExpirations) {
      throw new UnauthorizedException('Token expired or invalid');
    }

    // Check clearance
    const allowed = await this.checkClearance(
      tokenId,
      senderAddress,
      tokenType,
    );
    if (!allowed) {
      throw new UnauthorizedException('Not allowed');
    }

    this.logger.log('Access validation successful', { tokenId, senderAddress });
    return true;
  }
}
