import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { ethers } from 'ethers';
import { ContractFactoryService } from '../../shared-local/services/contract-factory.service';
import { ConfigService } from '@nestjs/config';

interface WithdrawFundsResponse {
  success: boolean;
  transactionHash?: string;
  blockNumber?: number;
  gasUsed?: string;
  message?: string;
  error?: string;
}

interface TokenExpirations {
  type: string;
  expiration: number[];
}

@Injectable()
export class BlockchainGatewayService {
  private readonly logger = new Logger(BlockchainGatewayService.name);

  private tradingManagement!: ethers.Contract;
  private dataAccessSubscription!: ethers.Contract;
  private dataAccessPayAsYouGo!: ethers.Contract;
  private dataAccessPayAsYouUse!: ethers.Contract;
  private offeringToken!: ethers.Contract;
  private paymentToken!: ethers.Contract;
  private wallet!: ethers.Wallet;

  constructor(
    private readonly contractFactory: ContractFactoryService,
    private readonly configService: ConfigService,
  ) {}

  async onModuleInit(): Promise<void> {
    this.logger.log('BlockchainGatewayService initializing...');

    try {
      await this.contractFactory.initialize();
      
      // Initialize contracts
      this.tradingManagement = this.contractFactory.getTradingManagement();
      this.dataAccessSubscription = this.contractFactory.getDataAccessSubscription();
      this.dataAccessPayAsYouGo = this.contractFactory.getDataAccessPayAsYouGo();
      this.dataAccessPayAsYouUse = this.contractFactory.getDataAccessPayAsYouUse();
      this.offeringToken = this.contractFactory.getOfferingToken();
      this.paymentToken = this.contractFactory.getPaymentToken();

      // Initialize wallet
      const privateKey = this.configService.getOrThrow<string>('PROVIDER_PRIVATE_KEY');
      this.wallet = new ethers.Wallet(privateKey, this.contractFactory.getProvider());

      this.logger.log('BlockchainGatewayService initialized');
    } catch (error) {
      this.logger.error('Failed to initialize BlockchainGatewayService', error);
      throw error;
    }
  }

  async getSubscriptionExpiration(userAddress: string, oid: string): Promise<number[]> {
    try {
      const rawExpirations = await this.tradingManagement.getAllDataAccessExpirationsSubscription(
        userAddress,
        oid,
      );
      return rawExpirations.map((bn: ethers.BigNumber) => bn.toNumber());
    } catch (error) {
      this.logger.error('Error fetching subscription expiration', { userAddress, oid, error });
      throw error;
    }
  }

  async getPAYGExpiration(userAddress: string, oid: string): Promise<number[]> {
    try {
      const rawExpirations = await this.tradingManagement.getAllDataAccessExpirationsPAYG(
        userAddress,
        oid,
      );
      return rawExpirations.map((bn: ethers.BigNumber) => bn.toNumber());
    } catch (error) {
      this.logger.error('Error fetching PAYG expiration', { userAddress, oid, error });
      throw error;
    }
  }

  async getPAYUExpiration(userAddress: string, oid: string): Promise<number[]> {
    try {
      const rawExpirations = await this.tradingManagement.getAllDataAccessExpirationsPAYU(
        userAddress,
        oid,
      );
      return rawExpirations.map((bn: ethers.BigNumber) => bn.toNumber());
    } catch (error) {
      this.logger.error('Error fetching PAYU expiration', { userAddress, oid, error });
      throw error;
    }
  }

  async getAllTokenExpirations(userAddress: string, oid: string): Promise<TokenExpirations[]> {
    try {
      const [subscriptionExp, paygExp, payuExp] = await Promise.all([
        this.getSubscriptionExpiration(userAddress, oid),
        this.getPAYGExpiration(userAddress, oid),
        this.getPAYUExpiration(userAddress, oid),
      ]);

      return [
        { type: 'SUB', expiration: subscriptionExp },
        { type: 'PAYG', expiration: paygExp },
        { type: 'PAYU', expiration: payuExp },
      ];
    } catch (error) {
      this.logger.error('Error fetching all token expiration times', { userAddress, oid, error });
      throw error;
    }
  }

  async checkClearance(assetId: string, userAddress: string, tokenType: string): Promise<boolean> {
    try {
      // Handle single address properly
      const addressArray = [userAddress];
      const assetArray = [assetId];

      let balances: ethers.BigNumber[];

      switch (tokenType) {
        case 'SUB':
          balances = await this.dataAccessSubscription.balanceOfBatch(addressArray, assetArray);
          break;
        case 'PAYG':
          balances = await this.dataAccessPayAsYouGo.balanceOfBatch(addressArray, assetArray);
          break;
        case 'PAYU':
          balances = await this.dataAccessPayAsYouUse.balanceOfBatch(addressArray, assetArray);
          break;
        default:
          throw new BadRequestException(`Unsupported tokenType: ${tokenType}`);
      }

      // Check if user has any balance
      return balances.some(balance => balance.gt(0));
    } catch (error) {
      this.logger.error('Error checking clearance', { assetId, userAddress, tokenType, error });
      throw error;
    }
  }

  async getOidFromDataAccessTokenId(tokenId: string): Promise<string> {
    try {
      const result = await this.offeringToken.getOidFromHash(tokenId);
      
      if (!result) {
        throw new BadRequestException(`No offer ID found for token: ${tokenId}`);
      }

      return result;
    } catch (error) {
      this.logger.error('Error getting OID from token ID', { tokenId, error });
      throw error;
    }
  }

  async getVolumeInKBPAYG(offerId: string): Promise<number> {
    try {
      const offerInfo = await this.offeringToken.getOfferIdInfo(offerId);
      const capVolumeStr = offerInfo.capVolume || '';
      
      if (!capVolumeStr) {
        throw new BadRequestException(`No volume cap found for offer: ${offerId}`);
      }

      const capVolume = parseInt(capVolumeStr.replace(/[^0-9]/g, ''), 10);
      
      if (isNaN(capVolume)) {
        throw new BadRequestException(`Invalid volume format for offer: ${offerId}`);
      }

      return capVolume;
    } catch (error) {
      this.logger.error('Error getting volume for PAYG', { offerId, error });
      throw error;
    }
  }

  async withdrawFunds(amount: string, fromAddress: string): Promise<WithdrawFundsResponse> {
    try {
      this.logger.log('Starting payment token withdrawal', { amount, fromAddress });

      const tokenWithSigner = this.paymentToken.connect(this.wallet);
      const decimals = await this.paymentToken.decimals();
      const amountInUnits = ethers.utils.parseUnits(amount, decimals);

      const tx = await tokenWithSigner.transferFrom(
        fromAddress,
        this.wallet.address,
        amountInUnits,
      );

      this.logger.log(`Withdraw tx submitted: ${tx.hash}`);

      const receipt = await tx.wait();
      this.logger.log(`Withdraw tx confirmed: ${receipt.transactionHash}`);

      if (receipt.status !== 1) {
        return {
          success: false,
          transactionHash: receipt.transactionHash,
          message: 'Transaction failed on blockchain',
        };
      }

      return {
        success: true,
        transactionHash: receipt.transactionHash,
        blockNumber: receipt.blockNumber,
        gasUsed: receipt.gasUsed?.toString(),
      };
    } catch (error) {
      this.logger.error('Error withdrawing funds', { amount, fromAddress, error });

      return {
        success: false,
        message: 'Payment token withdrawal failed',
        error: error instanceof Error ? error.message : 'Unknown error',
      };
    }
  }
}