import { Controller, Post, Query, HttpCode, HttpStatus } from '@nestjs/common';
import {
  ApiOperation,
  ApiTags,
  ApiUnauthorizedResponse,
  ApiBadRequestResponse,
  ApiOkResponse,
  ApiQuery,
} from '@nestjs/swagger';
import { BlockchainGatewayService } from '../blockchain-gateway/blockchain-gateway.service';
import { WithdrawFundsDto } from '../../dto/redeem-data-access.dto';

@ApiTags('Payment Processing')
@Controller('api/v1.0/payment')
export class PaymentTokenProcessorController {
  constructor(
    private readonly blockchainGatewayService: BlockchainGatewayService,
  ) {}

  @Post('withdraw-PAYG')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({
    summary: 'Process PAYG payment token withdrawal',
    description:
      'Withdraws payment tokens from user address to provider address for PAYG data access',
  })
  @ApiQuery({
    name: 'customerAddress',
    description: 'Ethereum address to withdraw tokens from',
    example: '0xf2d72bf781c11186553884574c01175c7d205822',
  })
  @ApiQuery({
    name: 'amount',
    description: 'Amount of tokens to withdraw',
    example: '1000',
  })
  @ApiOkResponse({
    description: 'Payment withdrawal successful',
    schema: {
      example: {
        success: true,
        transactionHash: '0xabcdef...',
        blockNumber: 12345,
        gasUsed: '21000',
      },
    },
  })
  @ApiUnauthorizedResponse({
    description:
      'Payment withdrawal failed due to insufficient balance or allowance',
  })
  @ApiBadRequestResponse({
    description: 'Invalid request parameters or blockchain error',
  })
  async withdrawPaymentTokens(@Query() dto: WithdrawFundsDto) {
    return await this.blockchainGatewayService.withdrawFunds(
      dto.amount,
      dto.customerAddress,
    );
  }
}
