import { Injectable, Logger } from '@nestjs/common';
import { ethers } from 'ethers';
import { firstValueFrom } from 'rxjs';
import { HttpService } from '@nestjs/axios';
import { ConfigService } from '@nestjs/config';
import { writeFileSync, existsSync, mkdirSync, readFileSync } from 'fs';
import { join } from 'path';

// Contract ABI imports - loaded dynamically to handle missing/empty files
// import TradingManagement from '../contracts/TradingManagement.json';
// import DataAccessSubscription from '../contracts/DataAccessSubscription.json';
// import DataAccessPayAsYouGo from '../contracts/DataAccessPayAsYouGo.json';
// import DataAccessPayAsYouUse from '../contracts/DataAccessPayAsYouUse.json';
// import OfferingToken from '../contracts/OfferingToken.json';
// import PaymentToken from '../contracts/PaymentToken.json';

export interface DataAccessContractAddresses {
  subscriptionAddress: string;
  paygAddress: string;
  payuAddress: string;
  offeringAddress: string;
  paymentAddress: string;
  tradingManagementAddress: string;
}

export interface ContractAddressesApiResponse {
  DataAccessSubscription: string;
  DataAccessPAYG: string;
  DataAccessPAYU: string;
  OfferingToken: string;
  PaymentToken: string;
  TradingManagement: string;
}

@Injectable()
export class ContractFactoryService {
  private readonly logger = new Logger(ContractFactoryService.name);
  public readonly provider: ethers.providers.JsonRpcProvider;
  private contractAddresses: DataAccessContractAddresses | null = null;
  private isInitialized = false;
  
  // Store ABIs dynamically loaded during initialization
  private tradingManagementABI: any[] = [];
  private dataAccessSubscriptionABI: any[] = [];
  private dataAccessPayAsYouGoABI: any[] = [];
  private dataAccessPayAsYouUseABI: any[] = [];
  private offeringTokenABI: any[] = [];
  private paymentTokenABI: any[] = [];

  constructor(
    private readonly httpService: HttpService,
    private readonly configService: ConfigService,
  ) {
    const rpcUrl = this.configService.getOrThrow<string>('BLOCKCHAIN_RPC_URL');
    this.provider = new ethers.providers.JsonRpcProvider(rpcUrl);
  }

  getProvider(): ethers.providers.JsonRpcProvider {
    return this.provider;
  }

  async initialize(): Promise<void> {
    if (this.isInitialized) {
      this.logger.debug('Contract addresses already initialized');
      return;
    }

    try {
      const tmApiUrl = this.configService.getOrThrow<string>('TM_API');
      const authToken = this.configService.getOrThrow<string>('JWT');

      // Fetch addresses first
      const addressesResponse = await firstValueFrom(
        this.httpService.get<ContractAddressesApiResponse>(
          `${tmApiUrl}/contract-info/addresses`,
          {
            headers: {
              Authorization: `Bearer ${authToken}`,
            },
          },
        ),
      );

      // Fetch ABIs for each contract
      await this.fetchAndSaveABIs(tmApiUrl, authToken);

      // Load ABIs from files into memory
      await this.loadABIsFromFiles();

      this.contractAddresses = this.mapApiResponseToContractAddresses(
        addressesResponse.data,
      );

      this.isInitialized = true;

      this.logger.log(
        'Successfully initialized contract addresses and ABIs',
        this.contractAddresses,
      );
    } catch (error) {
      this.logger.error('Failed to fetch contract addresses and ABIs', error);
      throw new Error(
        `Failed to initialize contract addresses and ABIs: ${error instanceof Error ? error.message : 'Unknown error'}`,
      );
    }
  }

  private mapApiResponseToContractAddresses(
    apiResponse: ContractAddressesApiResponse,
  ): DataAccessContractAddresses {
    return {
      subscriptionAddress: apiResponse.DataAccessSubscription,
      paygAddress: apiResponse.DataAccessPAYG,
      payuAddress: apiResponse.DataAccessPAYU,
      offeringAddress: apiResponse.OfferingToken,
      paymentAddress: apiResponse.PaymentToken,
      tradingManagementAddress: apiResponse.TradingManagement,
    };
  }

  private async loadABIsFromFiles(): Promise<void> {
    try {
      const contractsDir = join(__dirname, '..', 'contracts');
      
      // Contract file mappings
      const contractFiles = [
        { name: 'TradingManagement', property: 'tradingManagementABI' },
        { name: 'DataAccessSubscription', property: 'dataAccessSubscriptionABI' },
        { name: 'DataAccessPayAsYouGo', property: 'dataAccessPayAsYouGoABI' },
        { name: 'DataAccessPayAsYouUse', property: 'dataAccessPayAsYouUseABI' },
        { name: 'OfferingToken', property: 'offeringTokenABI' },
        { name: 'PaymentToken', property: 'paymentTokenABI' },
      ];

      for (const contractFile of contractFiles) {
        const filePath = join(contractsDir, `${contractFile.name}.json`);
        
        if (existsSync(filePath)) {
          try {
            const fileContent = readFileSync(filePath, 'utf8');
            const parsedABI = JSON.parse(fileContent);
            
            // Validate that the ABI is not empty
            if (Array.isArray(parsedABI) && parsedABI.length > 0) {
              (this as any)[contractFile.property] = parsedABI;
              this.logger.log(`Successfully loaded ABI for ${contractFile.name}`);
            } else {
              this.logger.warn(`ABI file for ${contractFile.name} is empty or invalid`);
              (this as any)[contractFile.property] = [];
            }
          } catch (parseError) {
            this.logger.warn(`Failed to parse ABI file for ${contractFile.name}`, parseError);
            (this as any)[contractFile.property] = [];
          }
        } else {
          this.logger.warn(`ABI file not found for ${contractFile.name} at ${filePath}`);
          (this as any)[contractFile.property] = [];
        }
      }
    } catch (error) {
      this.logger.error('Failed to load ABIs from files', error);
      throw new Error(
        `Failed to load ABIs from files: ${error instanceof Error ? error.message : 'Unknown error'}`,
      );
    }
  }

  private async fetchAndSaveABIs(
    tmApiUrl: string,
    authToken: string,
  ): Promise<void> {
    try {
      this.logger.log('Starting contract ABI update process');
      const contractsDir = join(__dirname, '..', 'contracts');

      // Ensure contracts directory exists
      if (!existsSync(contractsDir)) {
        mkdirSync(contractsDir, { recursive: true });
        this.logger.log(`Created contracts directory: ${contractsDir}`);
      }

      // Contract names to fetch ABIs for
      const contractNames = [
        'TradingManagementExecutorFacet',
        'DataAccessSubscription',
        'DataAccessPayAsYouGo',
        'DataAccessPayAsYouUse',
        'OfferingToken',
        'PaymentToken',
      ];

      // Fetch ABIs for each contract
      for (const contractName of contractNames) {
        try {
          const response = await firstValueFrom(
            this.httpService.get<any[]>(
              `${tmApiUrl}/contract-info/abi/by-name`,
              {
                headers: {
                  Authorization: `Bearer ${authToken}`,
                },
                params: {
                  contractName: contractName,
                },
              },
            ),
          );

          // Write ABI to JSON file
          const fileName = contractName === 'TradingManagementExecutorFacet' ? 'TradingManagement' : contractName;
          const filePath = join(contractsDir, `${fileName}.json`);
          const jsonContent = JSON.stringify(response.data, null, 2);

          // Check if file exists and log appropriately
          const fileExists = existsSync(filePath);
          writeFileSync(filePath, jsonContent, 'utf8');
          
          if (fileExists) {
            this.logger.log(`Successfully updated ABI for ${fileName}`);
          } else {
            this.logger.log(`Successfully created ABI file for ${fileName}`);
          }
        } catch (error) {
          this.logger.error(`Failed to fetch ABI for ${contractName}`, error);
          // Continue with other contracts even if one fails
        }
      }

      this.logger.log('Contract ABI update process completed');
    } catch (error) {
      this.logger.error('Failed to fetch and save ABIs', error);
      throw new Error(
        `Failed to fetch and save ABIs: ${error instanceof Error ? error.message : 'Unknown error'}`,
      );
    }
  }

  /**
   * Ensure contracts are initialized before use
   * @throws Error if contracts are not initialized
   */
  private ensureInitialized(): void {
    if (!this.isInitialized || !this.contractAddresses) {
      throw new Error(
        'ContractFactoryService not initialized. Call initialize() method first.',
      );
    }
  }

  /**
   * Create a contract instance with proper error handling
   */
  private createContract(
    address: string,
    abi: any,
    contractName: string,
  ): ethers.Contract {
    try {
      // Check if ABI is valid
      if (!abi || !Array.isArray(abi) || abi.length === 0) {
        throw new Error(`ABI for ${contractName} is empty or invalid. Contract ABIs may not have been loaded properly.`);
      }

      // Check if address is valid
      if (!address || !ethers.utils.isAddress(address)) {
        throw new Error(`Invalid contract address for ${contractName}: ${address}`);
      }

      return new ethers.Contract(address, abi, this.provider);
    } catch (error) {
      this.logger.error(`Failed to create ${contractName} contract`, error);
      throw new Error(
        `Failed to create ${contractName} contract: ${error instanceof Error ? error.message : 'Unknown error'}`,
      );
    }
  }

  getTradingManagement(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.tradingManagementAddress,
      this.tradingManagementABI,
      'TradingManagement',
    );
  }

  getDataAccessSubscription(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.subscriptionAddress,
      this.dataAccessSubscriptionABI,
      'DataAccessSubscription',
    );
  }

  getDataAccessPayAsYouGo(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.paygAddress,
      this.dataAccessPayAsYouGoABI,
      'DataAccessPayAsYouGo',
    );
  }

  getDataAccessPayAsYouUse(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.payuAddress,
      this.dataAccessPayAsYouUseABI,
      'DataAccessPayAsYouUse',
    );
  }

  getOfferingToken(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.offeringAddress,
      this.offeringTokenABI,
      'OfferingToken',
    );
  }

  getPaymentToken(): ethers.Contract {
    this.ensureInitialized();
    return this.createContract(
      this.contractAddresses!.paymentAddress,
      this.paymentTokenABI,
      'PaymentToken',
    );
  }

  /**
   * Check if the service is initialized
   */
  isServiceInitialized(): boolean {
    return this.isInitialized;
  }
}
