⚠️ **Reference Implementation**  
This project is provided as a reference implementation of a FAME-compliant Data Access Service.

# FAME Data Access Service (data-provider-poc)

The **data-provider-poc** is a lightweight public-facing proxy service that acts as a gateway between external customers and the internal **data-gate** service. It provides a clean API interface for token redemption while delegating all processing to the validator service.

## Overview

This service acts as a **proxy/gateway** that:

- Exposes a public API for data access token redemption
- Forwards validation requests to the internal data-gate
- Handles basic request/response formatting
- Provides API documentation via Swagger

The actual business logic (validation, payment processing, usage tracking) is handled by the **data-gate** service.

The data-provider-poc coordinates between:

- **External customers** - Token holders requesting data access
- **data-gate** - Internal validation and payment service
- **Trading Management API** - FAME platform token metadata

## Key Features

- **API Gateway** - Clean public interface for token redemption
- **Request Proxying** - Forwards requests to data-gate
- **Multi-Model Support** - SUB (Subscription), PAYG (Pay-As-You-Go), PAYU (Pay-As-You-Use)
- **Usage Tracking** - Monitors consumption and enforces volume limits

## Token Types

| Type     | Description               | Payment     | Volume Limits       |
| -------- | ------------------------- | ----------- | ------------------- |
| **SUB**  | Subscription-based access | Pre-paid    | None                |
| **PAYG** | Pay per KB consumed       | Per-request | Volume cap enforced |
| **PAYU** | Pay per usage session     | Pre-paid    | None                |

## API Reference

### POST `/api/redeem-data-access`

Redeems a Data Access Token and returns access credentials.

#### Process Flow

1. **Validate Access** - Verify token ownership via signature
2. **Check Token Info** - Fetch metadata from Trading Management API
3. **Enforce Limits** - Check volume constraints (PAYG only)
4. **Process Payment** - Execute blockchain payment via data-gate (PAYG only)
5. **Register Usage** - Track consumption
6. **Issue Credentials** - Return access details

#### Request Body

```json
{
  "tokenId": "osub123",
  "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "tokenType": "SUB" | "PAYG" | "PAYU",
  "signature": "0x1234567890abcdef...",
  "volumeInKB": "1234"  // Required for PAYG, optional for others
}
```

#### Responses

**✅ 200 Success**

_PAYG Token Response:_

```json
{
  "tokenId": "osub123",
  "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "tokenType": "PAYG",
  "validated": true,
  "accessDetails": {
    "description": "Dataset access granted",
    "accessCode": "1234",
    "bucketUrl": "example.com/dataset/1234",
    "usedVolumeInKB": "1234",
    "volumeLeft": 9876
  }
}
```

_SUB/PAYU Token Response:_

```json
{
  "tokenId": "osub123",
  "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "tokenType": "SUB",
  "validated": true,
  "accessDetails": {
    "description": "Dataset access granted",
    "accessCode": "1234",
    "bucketUrl": "example.com/dataset/1234"
  }
}
```

**❌ Error Responses**

| Status | Reason         | Description                                                                |
| ------ | -------------- | -------------------------------------------------------------------------- |
| `400`  | Bad Request    | Invalid request body, missing fields, or malformed data                    |
| `401`  | Unauthorized   | Invalid signature, expired token, insufficient balance, or volume exceeded |
| `500`  | Internal Error | Service unavailable or processing error                                    |

## Setup & Installation

### Prerequisites

- **Node.js** v20 or higher
- **npm** v9 or higher
- **Access** to FAME Trading Management API
- **Running instance** of data-gate service

### Installation Steps

1. **Install Dependencies**

   ```bash
   npm install
   ```

2. **Configure Environment**

   Create `.env` file:

   ```bash
   # Trading Management API
   TM_API=https://tm.fame-horizon.eu/api/v1.0
   JWT=your-jwt-token-for-tm-api

   # data-gate service
   DATA_ACCESS_VALIDATOR_API=http://localhost:8888/api/v1.0
   ```

3. **Start the Service**

```bash
npm run start
```

Service will be available at: `http://localhost:7777`

### Run with Docker (Optional)

```bash
docker build --network=host -f devops/Dockerfile -t data-provider-poc .
docker run --env-file .env -p 7777:7777 data-provider-poc
```

### API Documentation

Once running, access interactive API documentation at:

- **Swagger UI**: `http://localhost:7777/api`

## Dependencies & Integration

### Required Services

| Service                    | Purpose                               | Status Required       |
| -------------------------- | ------------------------------------- | --------------------- |
| **data-gate**              | Token validation & payment processing | ✅ Must be running    |
| **Trading Management API** | Token metadata                        | ✅ Must be accessible |

## Usage Examples

### Redeem Subscription Token

```bash
curl -X POST http://localhost:7777/api/redeem-data-access \
  -H "Content-Type: application/json" \
  -d '{
    "tokenId": "20728215483237468608613884902144932231752102498452206875205516149960849405151",
    "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
    "tokenType": "SUB",
    "signature": "0x1234567890abcdef..."
  }'
```

### Redeem PAYG Token

```bash
curl -X POST http://localhost:7777/api/redeem-data-access \
  -H "Content-Type: application/json" \
  -d '{
    "tokenId": "20728215483237468608613884902144932231752102498452206875205516149960849405151",
    "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
    "tokenType": "PAYG",
    "volumeInKB": "500",
    "signature": "0x1234567890abcdef..."
  }'
```

## Security Considerations

- **Signature Verification** - All requests must include valid cryptographic signatures
- **Token Validation** - Ownership verified through blockchain state
- **Payment Security** - PAYG transactions processed on-chain
- **Access Control** - Credentials are temporary and usage-tracked

## Troubleshooting

### Common Issues

| Issue                                   | Cause                              | Solution                                                      |
| --------------------------------------- | ---------------------------------- | ------------------------------------------------------------- |
| `401 Access validation failed`          | Invalid signature or expired token | Verify signature generation and token validity                |
| `401 Volume limit exceeded`             | PAYG token reached cap             | Check remaining volume or purchase additional capacity        |
| `500 Failed to fetch token information` | TM API unreachable                 | Verify `TM_API` URL and `JWT` token                           |
| `500 Failed to withdraw payment tokens` | data-gate payment error            | Check `DATA_ACCESS_VALIDATOR_API` and blockchain connectivity |

---

## Support

For implementation questions or issues:

- Review the [FAME Platform Documentation](https://docs.fame-horizon.eu)
- Check service logs for detailed error information
- Ensure all dependent services are operational
- In urgent issues contact tomasz.awramski@fujitsu.com

---

⚠️ **Important**: The **data-gate** service **must be running and reachable** at the URL specified in `DATA_ACCESS_VALIDATOR_API`, otherwise validation and payment processing will fail.
