import { ApiProperty } from '@nestjs/swagger';
import {
  IsEnum,
  IsEthereumAddress,
  IsNotEmpty,
  IsNumberString,
  IsString,
  ValidateIf,
} from 'class-validator';

// Enums and Types
export enum TokenType {
  SUB = 'SUB',
  PAYG = 'PAYG',
  PAYU = 'PAYU',
}

// Request DTOs
export class RedeemDataAccessDto {
  @ApiProperty({
    description: 'Token ID',
    example: 'osub123',
  })
  @IsString()
  @IsNotEmpty()
  tokenId!: string;

  @ApiProperty({
    description: 'Ethereum address of token owner',
    example: '0xf2d72bf781c11186553884574c01175c7d205822',
  })
  @IsEthereumAddress()
  @IsNotEmpty()
  senderAddress!: string;

  @ApiProperty({
    enum: TokenType,
    description: 'Token type: SUB, PAYG, or PAYU',
    example: TokenType.PAYG,
  })
  @IsEnum(TokenType, {
    message: `tokenType must be one of: ${Object.values(TokenType).join(', ')}`,
  })
  @IsNotEmpty()
  tokenType!: TokenType;

  @ApiProperty({
    description: 'Volume in KB (required for PAYG)',
    example: '1',
    required: false,
  })
  @ValidateIf((o) => o.tokenType === TokenType.PAYG)
  @IsNotEmpty({ message: 'volumeInKB is required for PAYG tokens' })
  @IsString()
  volumeInKB?: string;

  @ApiProperty({
    description: 'Authentication signature',
    example: '0x1234567890abcdef...',
  })
  @IsString()
  @IsNotEmpty()
  signature!: string;
}

export class WithdrawFundsDto {
  @ApiProperty({
    description: 'Customer Ethereum address to withdraw funds from',
    example: '0xf2d72bf781c11186553884574c01175c7d205822',
  })
  @IsEthereumAddress()
  @IsNotEmpty()
  customerAddress!: string;

  @ApiProperty({
    description: 'Amount to be withdrawn (in token units)',
    example: '1',
  })
  @IsNotEmpty()
  @IsNumberString()
  amount!: string;
}

// Response DTOs
export class AccessDetails {
  @ApiProperty({
    description: 'Description of the data access granted',
    example: 'Dataset access granted',
  })
  description!: string;

  @ApiProperty({
    description: 'Access code for the resource',
    example: '1234',
  })
  accessCode!: string;

  @ApiProperty({
    description: 'URL to access the data bucket or resource',
    example: 'example.com/dataset/1234',
  })
  bucketUrl!: string;

  @ApiProperty({
    description: 'Volume used in KB (present only for PAYG tokens)',
    example: '1',
    required: false,
  })
  usedVolumeInKB?: string;

  @ApiProperty({
    description: 'Remaining volume in KB (present only for PAYG tokens)',
    example: 999,
    required: false,
  })
  volumeLeft?: number;
}

export class RedeemDataAccessResponse {
  @ApiProperty({
    description: 'Token identifier',
    example: 'osub123',
  })
  tokenId!: string;

  @ApiProperty({
    description: 'Ethereum address of the token sender',
    example: '0xf2d72bf781c11186553884574c01175c7d205822',
  })
  senderAddress!: string;

  @ApiProperty({
    description: 'Type of the token',
    enum: TokenType,
    example: TokenType.PAYG,
  })
  tokenType!: string;

  @ApiProperty({
    description: 'Whether access was validated successfully',
    example: true,
  })
  validated!: boolean;

  @ApiProperty({
    description: 'Details about the granted access',
    type: AccessDetails,
  })
  accessDetails!: AccessDetails;
}
