/**
 * mock-usage-store.ts
 *
 * In-memory store for token usage records.
 *
 * NOTE: This is a mock — all data is lost on server restart.
 * Replace with a real database or persistent store for production use.
 */

export type UsageRecord = {
  tokenId: string;
  sender: string;
  tokenType: 'PAYU' | 'SUB' | 'PAYG';
  timestamp: number;
  volumeInKB?: string;
  downloads?: number;
};

export const usageStore: UsageRecord[] = [];

export function addUsage(record: UsageRecord) {
  const existing = usageStore.find(
    (r) =>
      r.tokenId === record.tokenId &&
      r.sender === record.sender &&
      r.tokenType === record.tokenType,
  );

  if (existing) {
    if (record.tokenType === 'PAYG' && record.volumeInKB) {
      // PAYG: Accumulate volume
      const existingVol = parseInt(existing.volumeInKB || '0', 10);
      const newVol = parseInt(record.volumeInKB, 10);
      existing.volumeInKB = (existingVol + newVol).toString();
      existing.timestamp = record.timestamp;
    } else if (record.tokenType === 'PAYU') {
      // PAYU: Increment usage count
      existing.downloads = (existing.downloads || 0) + 1;
      existing.timestamp = record.timestamp;
    } else {
      // SUB: Just update timestamp
      existing.timestamp = record.timestamp;
    }
  } else {
    // New record
    const newRecord = { ...record };
    if (record.tokenType === 'PAYU') {
      newRecord.downloads = 1; // Start counting from 1 if there is PAYU is accessed first time
    }
    usageStore.push(newRecord);
  }
}

export function getUsagesByAddress(address: string) {
  return usageStore.filter((r) => r.sender === address);
}

export function getUsedVolume(tokenId: string): number {
  const existing = usageStore.find(
    (r) =>
      r.tokenId === tokenId && r.sender === r.sender && r.tokenType === 'PAYG',
  );
  return existing?.volumeInKB ? parseInt(existing.volumeInKB, 10) : 0;
}

export function getUsedDownloads(tokenId: string, sender: string): number {
  const existing = usageStore.find(
    (r) =>
      r.tokenId === tokenId && r.sender === sender && r.tokenType === 'PAYU',
  );
  return existing?.downloads || 0;
}
