import { Controller, Post, Body, HttpCode, HttpStatus } from '@nestjs/common';
import {
  ApiOperation,
  ApiTags,
  ApiBody,
  ApiUnauthorizedResponse,
  ApiBadRequestResponse,
  ApiOkResponse,
  ApiInternalServerErrorResponse,
} from '@nestjs/swagger';
import {
  RedeemDataAccessDto,
  RedeemDataAccessResponse,
} from '../../dto/redeem-data-access.dto';
import { RedeemDataAccessService } from './redeem-data-access.service';

@ApiTags('Data Access Redemption')
@Controller('api')
export class RedeemDataAccessController {
  constructor(
    private readonly redeemDataAccessService: RedeemDataAccessService,
  ) {}

  @Post('redeem-data-access')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({
    summary: 'Redeem data access token',
    description: `
      Validates token ownership and provides access to data resources.
      
      **Process:**
      1. Validates access token and signature
      2. Checks token information and volume limits (for PAYG tokens)
      3. Processes payment withdrawal (for PAYG tokens)
      4. Registers token usage
      5. Returns access credentials
      
      **Token Types:**
      - **SUB** (Subscription): Time-based access with no volume limits
      - **PAYG** (Pay As You Go): Volume-based access with payment per KB
      - **PAYU** (Pay As You Use): Usage-based access model
    `,
  })
  @ApiBody({
    type: RedeemDataAccessDto,
    description:
      'Token redemption details including signature for verification',
  })
  @ApiOkResponse({
    description:
      'Access validated successfully. Returns asset access metadata.',
    type: RedeemDataAccessResponse,
    schema: {
      example: {
        tokenId: 'osub123',
        senderAddress: '0xf2d72bf781c11186553884574c01175c7d205822',
        tokenType: 'SUB',
        validated: true,
        accessDetails: {
          description: 'Dataset access granted',
          accessCode: '1234',
          bucketUrl: 'example.com/dataset/1234',
        },
      },
    },
  })
  @ApiUnauthorizedResponse({
    description: `
      Authentication or authorization failed:
      - Invalid signature
      - Token expired or not found
      - Insufficient token allowance (PAYG)
      - Volume limit exceeded (PAYG)
      - Payment withdrawal failed (PAYG)
    `,

    schema: {
      example: {
        statusCode: 401,
        message: 'Access validation failed',
        timestamp: '2025-06-09T18:39:49.123Z',
        path: '/api/redeem-data-access',
      },
    },
  })
  @ApiBadRequestResponse({
    description:
      'Invalid request body, missing required fields, or malformed data',

    schema: {
      example: {
        statusCode: 400,
        message: [
          'tokenId should not be empty',
          'senderAddress must be a valid Ethereum address',
        ],
        error: 'Bad Request',
        timestamp: '2025-06-09T18:39:49.123Z',
        path: '/api/redeem-data-access',
      },
    },
  })
  @ApiInternalServerErrorResponse({
    description: 'Internal server error during processing',

    schema: {
      example: {
        statusCode: 500,
        message: 'Internal server error',
        timestamp: '2025-06-09T18:39:49.123Z',
        path: '/api/redeem-data-access',
      },
    },
  })
  async redeemDataAccess(
    @Body() body: RedeemDataAccessDto,
  ): Promise<RedeemDataAccessResponse> {
    return await this.redeemDataAccessService.redeemDataAccess(body);
  }
}
