import { Injectable, Logger, UnauthorizedException } from '@nestjs/common';
import { HttpService } from '@nestjs/axios';
import { TokenUsageService } from './mock-token-usage.service';
import { RedeemDataAccessDto } from '../../dto/redeem-data-access.dto';
import { getUsedDownloads, getUsedVolume } from './mock-usage-store';
import { ConfigService } from '@nestjs/config';
import { firstValueFrom } from 'rxjs';

interface TokenInfoResponse {
  capVolume: string | null;
  capDownloads: string | null;
}

interface RedeemResponse {
  tokenId: string;
  senderAddress: string;
  tokenType: string;
  validated: boolean;
  accessDetails: AccessDetails;
}

interface AccessDetails {
  description: string;
  accessCode: string;
  bucketUrl: string;
  usedVolumeInKB?: string;
  volumeLeft?: number;
  downloadsLeft?: number;
}

interface WithdrawResponse {
  success: boolean;
  transactionHash?: string;
  blockNumber?: number;
  gasUsed?: string;
  message?: string;
  error?: string;
}

@Injectable()
export class RedeemDataAccessService {
  private logger = new Logger(RedeemDataAccessService.name);

  constructor(
    private readonly httpService: HttpService,
    private readonly tokenUsageService: TokenUsageService,
    private readonly configService: ConfigService,
  ) {}

  async redeemDataAccess(dto: RedeemDataAccessDto): Promise<RedeemResponse> {
    const { tokenId, senderAddress, tokenType } = dto;

    this.logger.log('Validating access', { dto });

    // Validate access with data-gate
    const validationResult = await this.validateAccess(dto);
    if (!validationResult) {
      throw new UnauthorizedException('Access validation failed');
    }

    // Get token information from TM
    const tokenInfo = await this.getTokenInfo(tokenId);

    // Handle volume validation for PAYG tokens
    const volumeDetails = this.handleVolumeValidation(
      tokenInfo,
      tokenType,
      tokenId,
      dto,
    );

    // Handle download validation for PAYU tokens
    const downloadDetails = this.handleDownloadValidation(
      tokenInfo,
      tokenType,
      tokenId,
      senderAddress,
    );

    // Process payment for PAYG tokens
    if (tokenType === 'PAYG' && dto.volumeInKB) {
      await this.withdrawPaymentTokens(senderAddress, dto.volumeInKB);
    }

    // Register token usage
    this.tokenUsageService.registerUsage(
      tokenId,
      senderAddress,
      tokenType,
      dto.volumeInKB,
    );

    // Return appropriate response
    return this.buildResponse(
      tokenId,
      senderAddress,
      tokenType,
      dto,
      { ...volumeDetails, ...downloadDetails }, // Merge both details
    );
  }

  private async validateAccess(dto: RedeemDataAccessDto): Promise<boolean> {
    try {
      const response = await firstValueFrom(
        this.httpService.get<boolean>(
          `${this.configService.getOrThrow(
            'DATA_ACCESS_VALIDATOR_API',
          )}/validate/access-token`,
          {
            params: {
              tokenId: dto.tokenId,
              senderAddress: dto.senderAddress,
              tokenType: dto.tokenType,
              signature: dto.signature,
            },
            headers: {
              Authorization: `Bearer ${this.configService.getOrThrow('JWT')}`,
            },
          },
        ),
      );

      const isValid = response.data;
      this.logger.debug('Validation response', isValid);

      if (!isValid) {
        this.logger.warn('Access validation failed', {
          tokenId: dto.tokenId,
          senderAddress: dto.senderAddress,
          validationResponse: isValid,
        });
      }

      return isValid;
    } catch (error) {
      this.logger.error('Error during access validation', {
        error: error instanceof Error ? error.message : 'Unknown error',
        tokenId: dto.tokenId,
        senderAddress: dto.senderAddress,
      });
      return false;
    }
  }

  private async getTokenInfo(tokenId: string): Promise<TokenInfoResponse> {
    try {
      const response = await firstValueFrom(
        this.httpService.get<TokenInfoResponse>(
          `${this.configService.getOrThrow(
            'TM_API',
          )}/data-access-info/get-data-access-token-info?tokenId=${tokenId}`,
          {
            headers: {
              Authorization: `Bearer ${this.configService.getOrThrow('JWT')}`,
            },
          },
        ),
      );

      return response.data;
    } catch (error) {
      this.logger.error('Error fetching token info', {
        error: error instanceof Error ? error.message : 'Unknown error',
        tokenId,
      });
      throw new UnauthorizedException('Failed to fetch token information');
    }
  }

  private handleVolumeValidation(
    tokenInfo: TokenInfoResponse,
    tokenType: string,
    tokenId: string,
    dto: RedeemDataAccessDto,
  ): { volumeLeft?: number } {
    if (tokenType !== 'PAYG' || !tokenInfo.capVolume) {
      return {};
    }

    const previouslyUsedVolumeKB = getUsedVolume(tokenId);
    const requestedVolumeKB = parseInt(dto.volumeInKB || '0', 10);
    const projectedTotalVolumeKB = previouslyUsedVolumeKB + requestedVolumeKB;

    const volumeLimitKB = parseInt(
      tokenInfo.capVolume.replace(/[^0-9]/g, ''),
      10,
    );

    const remainingVolumeKB = volumeLimitKB - projectedTotalVolumeKB;

    this.logger.debug('Volume validation check', {
      tokenId,
      previouslyUsedVolumeKB,
      requestedVolumeKB,
      projectedTotalVolumeKB,
      volumeLimitKB,
      remainingVolumeKB,
    });
    if (projectedTotalVolumeKB > volumeLimitKB) {
      this.logger.warn('Volume limit will be exceeded by this request', {
        tokenId,
        projectedTotalVolumeKB,
        volumeLimitKB,
        exceedBy: projectedTotalVolumeKB - volumeLimitKB,
      });
      throw new UnauthorizedException('Requested access exceeds volume limit');
    }

    return { volumeLeft: remainingVolumeKB };
  }

  private handleDownloadValidation(
    tokenInfo: TokenInfoResponse,
    tokenType: string,
    tokenId: string,
    senderAddress: string,
  ): { downloadsLeft?: number } {
    if (tokenType !== 'PAYU' || !tokenInfo.capDownloads) {
      return {};
    }

    const downloadsUsedToThisPoint = getUsedDownloads(tokenId, senderAddress);
    const newDownloadRequest = 1; // Each access counts as 1 download
    const totalDownloadsUsed = downloadsUsedToThisPoint + newDownloadRequest;

    // Parse download cap (remove non-numeric characters)
    const downloadsCap = parseInt(
      tokenInfo.capDownloads.replace(/[^0-9]/g, ''),
      10,
    );
    const leftDownloads = downloadsCap - totalDownloadsUsed;

    this.logger.debug('Download validation', {
      tokenId,
      senderAddress,
      downloadsUsedToThisPoint,
      newDownloadRequest,
      totalDownloadsUsed,
      downloadsCap,
      leftDownloads,
    });

    if (totalDownloadsUsed > downloadsCap) {
      this.logger.warn('Download limit exceeded', {
        tokenId,
        senderAddress,
        totalDownloadsUsed,
        downloadsCap,
        exceedBy: totalDownloadsUsed - downloadsCap,
      });
      throw new UnauthorizedException(
        'Requested access exceeds download limit',
      );
    }

    return { downloadsLeft: leftDownloads };
  }

  private buildResponse(
    tokenId: string,
    senderAddress: string,
    tokenType: string,
    dto: RedeemDataAccessDto,
    details: { volumeLeft?: number; downloadsLeft?: number },
  ): RedeemResponse {
    const baseResponse: RedeemResponse = {
      tokenId,
      senderAddress,
      tokenType,
      validated: true,
      accessDetails: {
        description: 'This is example',
        accessCode: '1234',
        bucketUrl: 'example.com/dataset/1234',
      },
    };

    // Add volume-specific details for PAYG tokens
    if (tokenType === 'PAYG') {
      baseResponse.accessDetails.usedVolumeInKB = dto.volumeInKB || '';
      if (details.volumeLeft !== undefined) {
        baseResponse.accessDetails.volumeLeft = details.volumeLeft;
      }
    }

    // Add download-specific details for PAYU tokens
    if (tokenType === 'PAYU') {
      if (details.downloadsLeft !== undefined) {
        baseResponse.accessDetails.downloadsLeft = details.downloadsLeft;
      }
    }

    return baseResponse;
  }

  private async withdrawPaymentTokens(
    customerAddress: string,
    amount: string,
  ): Promise<void> {
    try {
      this.logger.log('Withdrawing payment tokens', {
        customerAddress,
        amount,
      });

      const response = await firstValueFrom(
        this.httpService.post<WithdrawResponse>(
          `${this.configService.getOrThrow(
            'DATA_ACCESS_VALIDATOR_API',
          )}/payment/withdraw-PAYG`,
          null,
          {
            params: {
              amount,
              customerAddress,
            },
            headers: {
              accept: 'application/json',
              Authorization: `Bearer ${this.configService.getOrThrow('JWT')}`,
            },
          },
        ),
      );

      const withdrawResult = response.data;

      // Check if withdrawal was successful
      if (!withdrawResult.success) {
        this.logger.error('Payment token withdrawal failed', {
          customerAddress,
          amount,
          response: withdrawResult,
        });
        throw new UnauthorizedException(
          withdrawResult.message || 'Payment token withdrawal failed',
        );
      }

      this.logger.log('Payment tokens withdrawn successfully', {
        customerAddress,
        amount,
        transactionHash: withdrawResult.transactionHash,
        blockNumber: withdrawResult.blockNumber,
        gasUsed: withdrawResult.gasUsed,
      });
    } catch (error) {
      this.logger.error('Error during payment token withdrawal', {
        error: error instanceof Error ? error.message : 'Unknown error',
        customerAddress,
        amount,
      });

      // Re-throw UnauthorizedException as-is, wrap other errors
      if (error instanceof UnauthorizedException) {
        throw error;
      }

      throw new UnauthorizedException('Failed to withdraw payment tokens');
    }
  }
}
