import {
  Controller,
  Get,
  HttpException,
  HttpStatus,
  Post,
  Put,
  Query,
  Body,
  UseInterceptors,
  Logger,
} from '@nestjs/common';
import { ApiOperation, ApiTags, ApiResponse } from '@nestjs/swagger';

import {
  LoggerInterceptor,
  GetAssetBalance,
  ApproveOperatorToSpendDataAssetDto,
  GetIsDataAssetApprovedForOperatorDto,
  PurchaseAccessRightAndMineDto,
  DataAccessService,
} from '@api-shared';
@UseInterceptors(LoggerInterceptor)
@Controller('test-only/data-access')
@ApiTags('Testing (deprecated - for test purposes only)')
export class DataAccessDeprecatedController {
  constructor(private readonly dataAccessService: DataAccessService) {}

  logger = new Logger(DataAccessDeprecatedController.name);

  @Post('submissions/order')
  @ApiOperation({
    summary:
      'FOR TESTING - Executes submit purchasing order and mint it on chain. For the prod application we should just return unsigned transaction.',
  })
  @ApiResponse({
    status: 200,
    description: 'Purchasing order submitted successfully',
  })
  async testSubmitPurchasingOrder(
    @Body() dto: PurchaseAccessRightAndMineDto,
  ): Promise<string> {
    return this.dataAccessService.purchaseAccessRightAndMine(dto);
  }

  // @Put('revoke-data-access')
  // @ApiOperation({
  //   summary: 'Revoke (burn) an existing data asset token from a user.',
  // })
  // async burnAsset(@Query() dto: BurnAssetDto): Promise<string> {
  //   try {
  //     const txHash = await this.dataAccessService.burnAccessToken(
  //       dto.userAddress,
  //       dto.privateKey,
  //       dto.assetToBurnOwnerAddress,
  //       dto.assetId,
  //     );
  //     return `Data asset with assetId ${dto.assetId} burned by user ${dto.userAddress}. Transaction hash: ${txHash}`;
  //   } catch (error) {
  //     throw new HttpException(
  //       'Failed to burn data asset',
  //       HttpStatus.INTERNAL_SERVER_ERROR,
  //     );
  //   }
  // }

  @Put('approve-operator-to-spend-data-asset')
  @ApiOperation({
    summary:
      'FOR TESTING - It should be called by the end user using GUI / Metamask wallet. This method approve operator to spend/transfer data asset tokens on behalf of the user',
  })
  async approveOperatorToSpend(
    @Query() dto: ApproveOperatorToSpendDataAssetDto,
  ): Promise<string> {
    try {
      const txHash =
        await this.dataAccessService.approveOperatorToSpendDataAsset(
          dto.privateKey,
          dto.approved,
        );
      return `Approval for operator to spend data assets for user ${
        dto.userAddress
      } ${dto.approved ? 'succeeded' : 'revoked'}. Transaction hash: ${txHash}`;
    } catch (error) {
      throw new HttpException(
        'Failed to approve operator',
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    }
  }

  @Get('get-has-access-right')
  @ApiOperation({ summary: 'FOR TESTING - Get owned asset amount of a user' })
  async getAssetBalance(@Query() dto: GetAssetBalance): Promise<string> {
    try {
      const hasRight: boolean = await this.dataAccessService.getHasAccessRight(
        dto.userAddress,
        dto.assetId,
      );
      return `${dto.userAddress} access right status on asset id ${dto.assetId} is: ${hasRight}`;
    } catch (error) {
      throw new HttpException('Asset not found', HttpStatus.NOT_FOUND);
    }
  }

  @Get('get-is-data-asset-approved-for-operator')
  @ApiOperation({
    summary:
      'FOR TESTING - checks if the operator is approved to spend user data access tokens on behalf of the user',
  })
  async getIsDataAssetApprovedForOperator(
    @Query() dto: GetIsDataAssetApprovedForOperatorDto,
  ): Promise<boolean> {
    try {
      const isApproved =
        await this.dataAccessService.getIsDataAssetApprovedForOperator(
          dto.userAddress,
        );
      return isApproved;
    } catch (error) {
      throw new HttpException(
        'Approval for user on operator not found',
        HttpStatus.NOT_FOUND,
      );
    }
  }
}
