import {
  Controller,
  Post,
  Delete,
  Get,
  Body,
  UseInterceptors,
  Logger,
  HttpException,
  HttpStatus,
  Query,
  Param,
} from '@nestjs/common';
import {
  OfferingService,
  LoggerInterceptor,
  AddOfferDto,
  OfferintIdDto,
} from '@api-shared';
import { ApiBody } from '@nestjs/swagger';
import { ApiOperation, ApiTags, ApiResponse, ApiParam } from '@nestjs/swagger';
@UseInterceptors(LoggerInterceptor)
@Controller('tm/v1.0/offerings')
@ApiTags('Offering - private API')
export class OfferingController {
  private logger = new Logger(OfferingController.name);

  constructor(private readonly offeringService: OfferingService) {}

  @Post('/')
  @ApiOperation({
    summary:
      'Creates offering token that represents the offering, and mint it on chain. It is signed by the operator private key',
  })
  @ApiBody({ description: 'Add offering DTO', type: AddOfferDto })
  @ApiResponse({
    status: 201,
    description: 'Returns transaction hash if asset wass added successfully',
  })
  @ApiResponse({
    status: 409,
    description: 'Asset was already created - conflict',
  })
  @ApiResponse({ status: 500, description: 'Failed to add data asset' })
  async addAsset(@Body() dto: AddOfferDto): Promise<string> {
    const txHash = await this.offeringService.addAssetSigned(
      dto.assetid,
      dto.oid,
      dto.resource,
      dto.beneficiary,
      dto.price,
      dto.cap_duration,
      dto.cap_downloads,
      dto.cap_volume,
      dto.cds_target,
      dto.cds_sl,
    );
    this.logger.log(`Blockchain transaction hash: ${txHash}`);
    return txHash;
  }

  @ApiOperation({
    summary: 'Remove (burn) an offering token.',
  })
  @Delete('/:oid')
  @ApiParam({
    name: 'oid',
    description: 'The ID of the offering to be removed',
    required: true,
  })
  @ApiResponse({ status: 200, description: 'Offer removed successfully' })
  @ApiResponse({ status: 404, description: 'Offer token was not found' })
  async burnAsset(@Param() dto: OfferintIdDto): Promise<string> {
    const txHash = await this.offeringService.burnAsset(dto.oid);
    return txHash;
  }

  @Get('/exist')
  @ApiOperation({
    summary: 'Check if the offering exists (minted/added)',
  })
  async getAssetBalance(@Query() dto: OfferintIdDto): Promise<boolean> {
    try {
      const exists: boolean = await this.offeringService.getAssetExistance(
        dto.oid,
      );
      return exists;
    } catch (error) {
      throw new HttpException(
        'Getting asset existance error',
        HttpStatus.NOT_FOUND,
      );
    }
  }

  @Get('/uri')
  @ApiOperation({
    summary:
      'Returns the uri (link) that the offering is pointing to. Link should contain offering details',
  })
  async getTokenURI(@Query() dto: OfferintIdDto): Promise<string> {
    try {
      const uri: string = await this.offeringService.getTokenURI(dto.oid);
      return uri;
    } catch (error) {
      throw new HttpException('Getting asset URI error', HttpStatus.NOT_FOUND);
    }
  }
}
