import {
  Controller,
  Post,
  Body,
  UseInterceptors,
  Delete,
  Get,
  Param,
  Query,
} from '@nestjs/common';
import {
  GovernanceService,
  MintOrBurnPaymentTokenDto,
  LoggerInterceptor,
  PaymentTokenBalanceResponseDto,
  GetTradingAccount,
} from '@api-shared';
import {
  ApiOperation,
  ApiTags,
  ApiResponse,
  ApiParam,
  ApiOkResponse,
} from '@nestjs/swagger';

@UseInterceptors(LoggerInterceptor)
@Controller('tm/v1.0/payment-tokens')
@ApiTags('Payment Tokens - private API')
export class PaymentController {
  constructor(private readonly governanceService: GovernanceService) {}

  @Post('/mint')
  @ApiOperation({
    summary: 'Mint payment tokens and transfer to trading account',
  })
  @ApiResponse({
    status: 201,
    description: 'Payment tokens minted successfully',
  })
  async mintPaymentToken(@Body() dto: MintOrBurnPaymentTokenDto): Promise<any> {
    await this.governanceService.postTradingAccount(
      dto.tradingAccount,
      'MINT',
      dto.amount,
    );
    return `Payment token minted successfully. Amount: ${dto.amount}`;
  }
  @Delete('/burn')
  @ApiOperation({
    summary: 'Burn payment tokens from trading account',
  })
  @ApiResponse({
    status: 201,
    description: 'Payment tokens burned successfully',
  })
  async burnPaymentToken(@Body() dto: MintOrBurnPaymentTokenDto): Promise<any> {
    await this.governanceService.postTradingAccount(
      dto.tradingAccount,
      'BURN',
      dto.amount,
    );
    return `Payment token burned successfully. Amount: ${dto.amount}`;
  }

  @Get('/balances/accounts/:tid')
  @ApiOperation({ summary: 'Get trading account payment token balance' })
  @ApiParam({
    name: 'tid',
    description: 'The trading account ethereum address',
    required: true,
  })
  @ApiOkResponse({
    type: PaymentTokenBalanceResponseDto,
  })
  @ApiResponse({
    status: 200,
    description: 'Trading account balance retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve trading account',
  })
  async getTradingAccount(
    @Param('tid') tid: string,
    @Query() dto: GetTradingAccount,
  ): Promise<PaymentTokenBalanceResponseDto> {
    const amount = await this.governanceService.getTradingAccount(
      tid,
      dto.token,
    );
    return amount;
  }
}
