import { Injectable, HttpException, Logger } from '@nestjs/common';
import { Request } from 'express';
import * as jwtSimple from 'jwt-simple';
import * as dotenv from 'dotenv';
import axios from 'axios';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile, override: true });
@Injectable()
export class AuthService {
  private logger = new Logger(AuthService.name);
  async getJwtTokenPayloadFromRequest(
    request: Request,
  ): Promise<Record<string, any>> {
    let token: string = request.headers['Authorization'] as string;
    const jwtSecret = process.env.JWT_SECRET;

    if (!token && request.get) {
      token = request.get('Authorization') as string;
    }

    if (token) {
      const parts = token.split(' ');

      if (parts.length !== 2 || parts[0] !== 'Bearer' || !parts[1]) {
        throw new HttpException('Wrong token structure', 401);
      }

      const jwtToken = parts[1];

      try {
        const decoded = await this.decodeJWTToken(jwtToken, jwtSecret);
        return { userDetails: decoded };
      } catch (e) {
        if (e instanceof SyntaxError) {
          throw new HttpException('Invalid token', 401);
        }
        throw new HttpException(e.toString(), 401);
      }
    }
  }

  private async decodeJWTToken(token: string, secret: string): Promise<any> {
    if (process.env.AUTH_METHOD === 'LOCAL') {
      return jwtSimple.decode(token, secret);
    }
    if (process.env.AUTH_METHOD === 'ENDPOINT') {
      return this.decodeJWTByCallingGovEndpoint(token);
    }
    throw new Error('Unknown AUTH_METHOD');
  }

  private async decodeJWTByCallingGovEndpoint(token: string): Promise<any> {
    const body = {
      jwtToken: token,
    };
    const config = {
      headers: { 'Content-Type': 'application/json' },
    };
    try {
      const response = await axios.post(
        `${process.env.AUTH_URL}/verify-token`,
        body,
        config,
      );
      return response.data;
    } catch (error) {
      this.logger.error({
        ctx: 'decodeJWTByCallingGovEndpoint',
        message: `Verify token call error`,
        error,
      });
      throw error;
    }
  }
}
