import {
  Injectable,
  CanActivate,
  ExecutionContext,
  UnauthorizedException,
  HttpException,
} from '@nestjs/common';
import { AuthService } from '../authentication/auth.service';
import { Reflector } from '@nestjs/core';

@Injectable()
export class AuthGuard implements CanActivate {
  constructor(
    private readonly AuthService: AuthService,
    private reflector: Reflector,
  ) {}

  async canActivate(context: ExecutionContext): Promise<true | any> {
    const request = context.switchToHttp().getRequest();
    const token = this.extractTokenFromHeader(request);
    const allowUnauthorizedRequest = this.reflector.get<boolean>(
      'allowUnauthorizedRequest',
      context.getHandler(),
    );
    if (allowUnauthorizedRequest) {
      return Promise.resolve(true);
    }

    if (!token) {
      throw new UnauthorizedException();
    }

    try {
      const payload =
        await this.AuthService.getJwtTokenPayloadFromRequest(request);
      request['userDetails'] = payload;
    } catch (error) {
      if (error instanceof HttpException) {
        throw error;
      }
      throw new HttpException('Unauthorized', 401);
    }
    return Promise.resolve(true);
  }

  private extractTokenFromHeader(request: any): string | undefined {
    const [type, token] = request.headers.authorization?.split(' ') ?? [];
    return type === 'Bearer' ? token : undefined;
  }
}
export { AuthService };
