import { Controller, Get, Query, Param, UseInterceptors } from '@nestjs/common';
import {
  GovernanceService,
  GetTradingAccount,
  LoggerInterceptor,
  PaymentTokenBalanceResponseDto,
  GetContractABIByNameDto,
  GetDiamondsInfoByNameDto,
} from '@api-shared';
import {
  ApiOperation,
  ApiTags,
  ApiResponse,
  ApiParam,
  ApiOkResponse,
  ApiBearerAuth,
} from '@nestjs/swagger';
import {
  ContractAddresses,
  getContractABI,
  getContractAddresses,
  readLastDeployedDiamondInfo,
} from '@contracts';
import { AllowUnauthorizedRequest } from '../../guards/allow-unathorized-request.decorator';

@ApiBearerAuth()
@UseInterceptors(LoggerInterceptor)
@Controller('contract-info')
@ApiTags('Contract Information - public API')
export class ContractInfoController {
  constructor(private readonly governanceService: GovernanceService) {}

  @AllowUnauthorizedRequest()
  @Get('/addresses/data-access')
  @ApiOperation({
    summary: 'Retrieves the latest deployed data access contract addresses.',
  })
  @ApiResponse({
    status: 200,
    description: 'Contracts Addresses retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve contracts Addresses',
  })
  async getDataAccessContractsAddresses(): Promise<any> {
    try {
      const contractAddresses: ContractAddresses = getContractAddresses(
        process.env.HARDHAT_NETWORK,
      );
      //cuts "contractAddress" from contract names
      const clearedObject = Object.fromEntries(
        Object.entries(contractAddresses).map(([key, value]) => [
          key.replace('contractAddress', ''),
          value,
        ]),
      );

      // Filter out only the data access contracts
      const dataAccessContracts = Object.fromEntries(
        Object.entries(contractAddresses).filter(([key]) =>
          key.startsWith('contractAddressDataAccess'),
        ),
      );

      return dataAccessContracts;
    } catch (error) {
      console.log(error);
    }
  }

  @Get('/addresses')
  @ApiOperation({
    summary: 'Retrieves the latest deployed contract addresses in the system.',
  })
  @ApiResponse({
    status: 200,
    description: 'Contracts Addresses retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve contracts Addresses',
  })
  async getContractsAddresses(): Promise<any> {
    try {
      const contractAddresses = getContractAddresses(
        process.env.HARDHAT_NETWORK,
      );

      //cuts "contractAddress" from contract names
      const clearedObject = Object.fromEntries(
        Object.entries(contractAddresses).map(([key, value]) => [
          key.replace('contractAddress', ''),
          value,
        ]),
      );

      return clearedObject;
    } catch (error) {
      console.log(error);
    }
  }

  @Get('/addresses/diamond-info')
  @ApiOperation({
    summary: 'Fetches metadata for a specific Diamond contract.',
  })
  @ApiResponse({
    status: 200,
    description: 'Diamond info retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve diamond info',
  })
  async getDiamondsInfo(@Query() dto: GetDiamondsInfoByNameDto): Promise<any> {
    try {
      return readLastDeployedDiamondInfo(
        process.env.HARDHAT_NETWORK,
        dto.diamondName,
      );
    } catch (error) {
      return `Diamond info not found.`;
    }
  }

  @Get('/abi/by-name')
  @ApiOperation({ summary: 'Get contract ABI by contract name' })
  @ApiResponse({
    status: 200,
    description: 'Contract ABI retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve contract ABI',
  })
  async getContractAbi(@Query() dto: GetContractABIByNameDto): Promise<any> {
    try {
      return getContractABI(dto.contractName);
    } catch (error) {
      return `Contract ${dto.contractName} not found.`;
    }
  }

  @Get('/payment-balances/accounts/:tid')
  @ApiOperation({ summary: 'Get trading account payment token balance' })
  @ApiParam({
    name: 'tid',
    description: 'The trading account ethereum address',
    required: true,
  })
  @ApiOkResponse({
    type: PaymentTokenBalanceResponseDto,
  })
  @ApiResponse({
    status: 200,
    description: 'Trading account balance retrieved successfully',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve trading account',
  })
  async getTradingAccount(
    @Param('tid') tid: string,
    @Query() dto: GetTradingAccount,
  ): Promise<PaymentTokenBalanceResponseDto> {
    const amount = await this.governanceService.getTradingAccount(
      tid,
      dto.token,
    );
    return amount;
  }
}
