import {
  Controller,
  Get,
  Query,
  UseInterceptors,
  HttpException,
  HttpStatus,
} from '@nestjs/common';
import {
  LoggerInterceptor,
  DataAccessService,
  UserAddressOfferingIdDto,
  CheckClearanceReq,
  OfferingService,
  FullOfferInfoDto,
} from '@api-shared';
import {
  ApiOperation,
  ApiTags,
  ApiResponse,
  ApiOkResponse,
  ApiBearerAuth,
  ApiQuery,
} from '@nestjs/swagger';

import { AllowUnauthorizedRequest } from '../../guards/allow-unathorized-request.decorator';

@ApiBearerAuth()
@UseInterceptors(LoggerInterceptor)
@Controller('data-access-info')
@ApiTags('Data Access Info - public API')
export class DataAccessInfoController {
  constructor(
    private readonly dataAccessService: DataAccessService,
    private readonly offeringService: OfferingService,
  ) {}

  @Get('check-clearance')
  @ApiOperation({
    summary:
      'Returns whether the given asset id is owned by given trading account(s).',
  })
  @ApiQuery({
    name: 'assetId',
    description: 'The asset ID (string index) to check clearance for',
    required: true,
  })
  @ApiQuery({
    name: 'addresses',
    description: 'The addresses of the trading accounts',
    required: true,
    isArray: true,
  })
  @ApiQuery({
    name: 'tokenType',
    description: 'The type of the token',
    required: true,
  })
  @ApiOkResponse({
    description: 'Clearance checked successfully',
  })
  @ApiResponse({ status: 500, description: 'Failed to check clearance' })
  async checkClearance(@Query() dto: CheckClearanceReq): Promise<boolean> {
    return this.dataAccessService.checkClearance(
      dto.assetId,
      dto.addresses,
      dto.tokenType,
    );
  }

  @Get('/subscription-last-expiration')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves the last subscription data access expiration time as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description:
      'Retrieves last subscription data access expiration time. Zero is returned if non-existent.',
  })
  @ApiResponse({
    status: 500,
    description:
      'Failed to retrieve last subscription data access expiration time.',
  })
  async getLastSubscriptionExpirationTime(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<number> {
    return this.dataAccessService.getLastDataAccessSubscriptionExpirationTime(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('/subscription-all-expirations')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves all subscription data access expiration time as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description:
      'Retrieves array of all subscription data access expiration times.',
  })
  @ApiResponse({
    status: 500,
    description:
      'Failed to retrieve all subscription data access expiration times.',
  })
  async getAllSubscriptionExpirationTimes(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<any> {
    return this.dataAccessService.getAllDataAccessExpirationsSubscription(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('/PAYU-last-expiration')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves the last PAYU data access expiration time as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description:
      'Retrieves last PAYU data access expiration. Zero is returned if non-existent.',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve PAYU data access expiration time',
  })
  async getLastPAYUExpirationTime(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<number> {
    return this.dataAccessService.getLastDataAccessPAYUExpirationTime(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('/PAYU-all-expirations')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves all PAYU data access expiration times as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description: 'Retrieves array of all PAYU data access expirations.',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve PAYU data access expiration time',
  })
  async getAllPAYUExpirationTimes(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<any> {
    return this.dataAccessService.getAllDataAccessExpirationsPAYU(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('/PAYG-last-expiration')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves the last PAYG data access expiration time as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description:
      'Retrieves last PAYG data access expiration time. Zero is returned if non-existent.',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve PAYG data access expiration time.',
  })
  async getLastPAYGExpirationTime(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<number> {
    return this.dataAccessService.getLastDataAccessPAYGExpirationTime(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('/PAYG-all-expirations')
  @ApiOperation({
    summary:
      'Given user address and offering id for data, it retrieves all PAYG data access expiration times as unixtime.',
  })
  @ApiOkResponse({
    type: String,
  })
  @ApiResponse({
    status: 200,
    description: 'Retrieves array of all PAYG data access expiration times.',
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve all PAYG data access expiration times',
  })
  async getAllPAYGExpirationTimes(
    @Query() dto: UserAddressOfferingIdDto,
  ): Promise<any> {
    return this.dataAccessService.getAllDataAccessExpirationsPAYG(
      dto.userAddress,
      dto.tokenId,
    );
  }

  @Get('get-offer-info')
  @ApiResponse({
    status: 200,
    description: 'Retrieves array of all PAYG data access expiration times.',
    type: FullOfferInfoDto,
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve all PAYG data access expiration times',
  })

  //TODO add types
  async getOfferInfo(@Query('oid') oid: string): Promise<FullOfferInfoDto> {
    const offerInfo = await this.offeringService.getFullOfferIdInfo(oid);
    return offerInfo;
  }

  @Get('get-data-access-token-info')
  @ApiOperation({
    summary: 'Return dataAccessToken info',
  })
  @ApiOkResponse({
    description: 'Return dataAccessToken info',
  })
  async getDataAccessTokenInfo(
    @Query('tokenId') tokenId: string,
  ): Promise<Record<string, unknown>> {
    try {
      const offerInfo =
        await this.offeringService.getDataAccessTokenInfo(tokenId);

      return {
        dataAccessPrice: offerInfo[0].toString(),
        capDuration: offerInfo[1].toString(),
        capDownloads: offerInfo[2],
        capVolume: offerInfo[3],
        cdsTarget: offerInfo[4],
        cdsSl: offerInfo[5],
        oid: offerInfo[6],
      };
    } catch (error) {
      throw new HttpException(error, HttpStatus.NOT_FOUND);
    }
  }

  @Get('get-tokenid-from-offerid')
  @ApiOperation({
    summary: 'Return tokenid from offerid',
  })
  async getAssetId(@Query('offerId') offerId: string): Promise<any> {
    try {
      const assetId = await this.offeringService.getIDHash(offerId);
      const decimalString = assetId.toString();
      return decimalString;
    } catch (error) {
      throw new HttpException('Getting asset URI error', HttpStatus.NOT_FOUND);
    }
  }
}
