import {
  Controller,
  Get,
  Query,
  UseInterceptors,
  Request,
} from '@nestjs/common';
import {
  ApiBearerAuth,
  ApiOperation,
  ApiResponse,
  ApiTags,
} from '@nestjs/swagger';

import {
  PaymentService,
  LoggerInterceptor,
  GetPaymentAllowanceOnOperatorDto,
  IncreaseAllowanceDto,
  PaymentAllowanceQueryDto,
} from '@api-shared';

@ApiBearerAuth()
@UseInterceptors(LoggerInterceptor)
@Controller('payment')
@ApiTags('Payment - public API')
export class PaymentController {
  constructor(private readonly paymentService: PaymentService) {}

  @Get('approve-operator-to-spend-payment-tokens')
  @ApiOperation({
    summary:
      'Generates an unsigned transaction to approve unlimited spending of FDE tokens by the operator.',
  })
  @ApiResponse({
    status: 200,
    description:
      'Returns unsigned transaction for spending unlimited amount of FDE tokes',
    type: String,
  })
  async approveToSpendOperator(): Promise<any> {
    return this.paymentService.approveToSpendUnlimitedPaymentToken();
  }

  @Get('get-payment-allowance-on-operator')
  @ApiOperation({
    summary:
      'Fetches the approved spending limit of Payment Tokens for a specified operator.',
  })
  @ApiResponse({
    status: 200,
    description: 'Returns approved amount for spending Payment tokens',
    type: Number,
  })
  async getPaymentTokenAllowance(
    @Query() dto: GetPaymentAllowanceOnOperatorDto,
  ): Promise<number> {
    return this.paymentService.getPaymentTokenAllowanceOnOperator(
      dto.userAddress,
    );
  }

  @Get('get-increase-allowance-tx')
  @ApiOperation({
    summary:
      'Generates an unsigned transaction to approve a specific amount of FDE tokens to any address.',
  })
  @ApiResponse({
    status: 200,
    description:
      'Returns unsigned transaction to allow spending a specific amount of FDE tokens',
    type: String,
  })
  async getIncreaseAllowanceTransaction(
    @Query() dto: IncreaseAllowanceDto,
  ): Promise<any> {
    return this.paymentService.getIncreaseAllowanceTx(
      dto.toApproveAddress,
      dto.amountToAllow,
    );
  }

  @Get('get-payment-allowance-on-address')
  @ApiOperation({
    summary: 'Returns how much the address can withdraw from the user.',
  })
  @ApiResponse({
    status: 200,
    description: 'Returns allowance of specified address on users tokens',
    type: Number,
  })
  async getPaymentTokenAllowanceOnAddress(
    @Query() query: PaymentAllowanceQueryDto,
  ): Promise<number> {
    return this.paymentService.getPaymentTokenAllowance(
      query.userAddress,
      query.allowanceCheckAddress,
    );
  }
}
